<?php

namespace Drupal\Tests\revision_manager\Kernel;

use Drupal\Core\Entity\RevisionableInterface;

/**
 * Tests the Amount plugin of Revision Manager for different entity types.
 *
 * @group revision_manager
 */
class AmountPluginTest extends RevisionManagerTestBase {

  /**
   * Test amount plugin on nodes.
   */
  public function testAmountPluginNode(): void {
    $this->configureRevisionManager('node', 'amount', TRUE, ['amount' => 5]);
    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Node Amount Test', 'status' => 1]);

    $this->assertInstanceOf(RevisionableInterface::class, $entity);
    $this->createRevisions($entity, 10);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify the node has exactly 5 revisions remaining.
    $this->assertRevisionCount($entity, 5);
  }

  /**
   * Test amount plugin on block content.
   */
  public function testAmountPluginBlockContent(): void {
    $this->configureRevisionManager('block_content', 'amount', TRUE, ['amount' => 3]);
    $entity = $this->createEntity('block_content', ['type' => 'basic', 'info' => 'BlockTest', 'status' => 1]);

    $this->assertInstanceOf(RevisionableInterface::class, $entity);
    $this->createRevisions($entity, 5);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify the block content has exactly 3 revisions remaining.
    $this->assertRevisionCount($entity, 3);
  }

  /**
   * Test amount plugin on taxonomy terms.
   */
  public function testAmountPluginTaxonomyTerm(): void {
    $this->configureRevisionManager('taxonomy_term', 'amount', TRUE, ['amount' => 8]);
    $entity = $this->createEntity('taxonomy_term', ['vid' => 'tags', 'name' => 'TermTest', 'status' => 1]);

    $this->assertInstanceOf(RevisionableInterface::class, $entity);
    $this->createRevisions($entity, 10);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify the taxonomy term has exactly 8 revisions remaining.
    $this->assertRevisionCount($entity, 8);
  }

  /**
   * Test amount plugin on media entities.
   */
  public function testAmountPluginMedia(): void {
    $this->configureRevisionManager('media', 'amount', TRUE, ['amount' => 10]);
    $media = $this->createEntity('media', ['bundle' => 'document', 'name' => 'Media Test Document', 'status' => 1]);

    $this->assertInstanceOf(RevisionableInterface::class, $media);
    $this->createRevisions($media, 15);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify the media has exactly 10 revisions remaining.
    $this->assertRevisionCount($media, 10);
  }

  /**
   * Test amount plugin on group entities.
   */
  public function testAmountPluginGroup(): void {
    $this->configureRevisionManager('group', 'amount', TRUE, ['amount' => 5]);
    $entity = $this->createEntity('group', ['type' => 'test_group_type', 'label' => 'Group Test Amount Plugin']);

    $this->assertInstanceOf(RevisionableInterface::class, $entity);
    $this->createRevisions($entity, 10);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify the group has exactly 5 revisions remaining.
    $this->assertRevisionCount($entity, 5);
  }

  /**
   * Test amount plugin on menu link content entities.
   */
  public function testAmountPluginMenuLinkContent(): void {
    $this->configureRevisionManager('menu_link_content', 'amount', TRUE, ['amount' => 4]);
    $entity = $this->createEntity('menu_link_content', [
      'title' => 'Menu Link Amount Test',
      'link' => ['uri' => 'internal:/'],
      'menu_name' => 'main',
    ]);
    $this->assertInstanceOf(RevisionableInterface::class, $entity);

    $this->createRevisions($entity, 8);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify exactly 4 revisions remain.
    $this->assertRevisionCount($entity, 4);
  }

  /**
   * Test plugin disabled does nothing.
   */
  public function testAmountPluginDisabled(): void {
    $this->configureRevisionManager('node', 'amount', FALSE, ['amount' => 2]);
    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Disabled Amount Plugin Test', 'status' => 1]);

    $this->assertInstanceOf(RevisionableInterface::class, $entity);
    $this->createRevisions($entity, 4);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify the node has 5 revisions remaining since the plugin is disabled.
    $this->assertRevisionCount($entity, 5);
  }

  /**
   * Test entity type disabled prevents deletions.
   */
  public function testEntityDisabled(): void {
    $this->configureRevisionManager('node', 'amount', TRUE, ['amount' => 2]);
    $config = $this->config('revision_manager.settings');
    $enabled = $config->get('enabled_entities');
    $enabled['node'] = FALSE;
    $config->set('enabled_entities', $enabled)->save();
    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Disabled Entity Test', 'status' => 1]);

    $this->assertInstanceOf(RevisionableInterface::class, $entity);
    $this->createRevisions($entity, 5);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify the node has 6 revisions remaining since the entity is disabled.
    $this->assertRevisionCount($entity, 6);
  }

  /**
   * Test amount plugin bundle override.
   */
  public function testAmountPluginBundleOverride(): void {
    $this->configureRevisionManager('node', 'amount', TRUE, ['amount' => 5]);
    $bundle = $this->loadBundle('node_type', 'page');
    $bundle->setThirdPartySetting(
      'revision_manager',
      'amount',
      ['status' => TRUE, 'settings' => ['amount' => 2]],
    );
    $bundle->save();
    $this->container->get('plugin.manager.revision_manager')->clearCachedDefinitions();

    // Create a node of type 'page' with 10 revisions.
    $node = $this->createEntity('node', ['type' => 'page', 'title' => 'Override Test', 'status' => 1]);
    $this->assertInstanceOf(RevisionableInterface::class, $node);
    $this->createRevisions($node, 10);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify only 2 revisions remain for the 'page' bundle.
    $this->assertRevisionCount($node, 2);
  }

  /**
   * Test unpublished revisions are not deleted.
   */
  public function testUnpublishedRevisionsAreNotDeleted(): void {
    $this->configureRevisionManager('node', 'amount', TRUE, ['amount' => 3]);
    $node = $this->createEntity('node', ['type' => 'page', 'title' => 'Publish Test', 'status' => 1]);

    $this->assertInstanceOf(RevisionableInterface::class, $node);

    $this->createRevisions($node, 5);
    $this->createRevisions($node, 4, ['status' => 0]);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify 3 published kept + 4 drafts.
    $this->assertRevisionCount($node, 7);
  }

  /**
   * Test amount greater than revisions does nothing.
   */
  public function testAmountGreaterThanRevisionsDoesNothing(): void {
    $this->configureRevisionManager('node', 'amount', TRUE, ['amount' => 20]);
    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Greater Than Test', 'status' => 1]);

    $this->assertInstanceOf(RevisionableInterface::class, $entity);
    $this->createRevisions($entity, 5);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify the node has 6 revisions.
    $this->assertRevisionCount($entity, 6);
  }

  /**
   * Test zero amount keeps only active revision.
   */
  public function testZeroAmountKeepsOnlyActive(): void {
    $this->configureRevisionManager('node', 'amount', TRUE, ['amount' => 0]);
    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Zero Test', 'status' => 1]);

    $this->assertInstanceOf(RevisionableInterface::class, $entity);

    $this->createRevisions($entity, 4);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify the current active revision is the only one remaining.
    $this->assertRevisionCount($entity, 1);
  }

  /**
   * Test amount plugin with a large number of revisions.
   */
  public function testAmountPluginWithManyRevisions(): void {
    $this->configureRevisionManager('node', 'amount', TRUE, ['amount' => 5]);
    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Many Revisions Test', 'status' => 1]);

    $this->assertInstanceOf(RevisionableInterface::class, $entity);

    $this->createRevisions($entity, 1000);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify that is matches the amount.
    $this->assertRevisionCount($entity, 5);
  }

}
