<?php

namespace Drupal\Tests\revision_manager\Kernel;

use Drupal\Core\Entity\RevisionableInterface;

/**
 * Tests combined Age + Amount plugins of Revision Manager.
 *
 * @group revision_manager
 */
class IntegrationPluginTest extends RevisionManagerTestBase {

  /**
   * Verify that both Age and Amount plugins work together on nodes.
   */
  public function testAgeAndAmountPluginsNode(): void {
    $this->configureRevisionManager('node', 'age', TRUE, ['age' => 6]);
    $this->configureRevisionManager('node', 'amount', TRUE, ['amount' => 3]);

    // Create the base node.
    $node = $this->createEntity('node', ['type' => 'page', 'title' => 'Age / Amount Integration Test', 'status' => 1]);
    $this->assertInstanceOf(RevisionableInterface::class, $node);

    $cutoff = (new \DateTimeImmutable())->modify('-6 months')->getTimestamp();

    // Add 5 revisions older than the cutoff and 6 newer ones.
    $this->createRevisionsAroundCutoff($node, 5, 6, $cutoff, self::ONE_DAY * 10, self::ONE_DAY);
    $this->processRevisionQueue(self::QUEUE_ID);

    // With intersection logic, only 5 revisions are deleted.
    $this->assertRevisionCount($node, 7);
  }

  /**
   * Verify unpublished revisions remain when both plugins run together.
   */
  public function testAgeAndAmountSkipUnpublished(): void {
    $this->configureRevisionManager('node', 'age', TRUE, ['age' => 6]);
    $this->configureRevisionManager('node', 'amount', TRUE, ['amount' => 2]);

    $node = $this->createEntity('node', [
      'type' => 'page',
      'title' => 'Age / Amount Unpublished Integration Test',
      'status' => 1,
    ]);
    $this->assertInstanceOf(RevisionableInterface::class, $node);

    $this->createRevisions($node, 3);
    $this->createRevisions($node, 2, ['status' => 0]);

    $this->processRevisionQueue(self::QUEUE_ID);

    // Age returns empty (all recent), Amount returns some.
    // Intersection = empty.
    // All 6 revisions remain: 1 original + 3 published + 2 unpublished.
    $this->assertRevisionCount($node, 6);
  }

  /**
   * Verify empty Age intersection with non-empty Amount deletes nothing.
   */
  public function testAgeEmptyAmountNonEmpty(): void {
    $this->configureRevisionManager('node', 'age', TRUE, ['age' => 6]);
    $this->configureRevisionManager('node', 'amount', TRUE, ['amount' => 2]);

    $node = $this->createEntity('node', [
      'type' => 'page',
      'title' => 'Age Empty / Amount Non-empty Test',
      'status' => 1,
    ]);
    $this->assertInstanceOf(RevisionableInterface::class, $node);

    // All revisions are recent, so Age returns empty.
    $this->createRevisions($node, 5);

    $this->processRevisionQueue(self::QUEUE_ID);

    // Age empty, Amount non-empty. Intersection = empty. All 6 remain.
    $this->assertRevisionCount($node, 6);
  }

  /**
   * Verify empty Amount intersection with non-empty Age deletes nothing.
   */
  public function testAmountEmptyAgeNonEmpty(): void {
    $this->configureRevisionManager('node', 'amount', TRUE, ['amount' => 10]);
    $this->configureRevisionManager('node', 'age', TRUE, ['age' => 6]);

    $node = $this->createEntity('node', [
      'type' => 'page',
      'title' => 'Amount Empty / Age Non-empty Test',
      'status' => 1,
    ]);
    $this->assertInstanceOf(RevisionableInterface::class, $node);

    $cutoff = (new \DateTimeImmutable())->modify('-6 months')->getTimestamp();

    // All revisions are old, so Age returns candidates.
    // Amount keeps 10, returns empty.
    $this->createRevisionsAroundCutoff($node, 5, 0, $cutoff, self::ONE_DAY * 10, self::ONE_DAY);

    $this->processRevisionQueue(self::QUEUE_ID);

    // Amount empty, Age non-empty. Intersection = empty. All 6 remain.
    $this->assertRevisionCount($node, 6);
  }

}
