<?php

namespace Drupal\Tests\revision_manager\Traits;

use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Entity\RevisionableStorageInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Entity\EntityChangedInterface;
use Drupal\Core\Queue\QueueWorkerInterface;
use Drupal\Core\Queue\QueueInterface;

/**
 * Provides helper methods for testing the revision manager module.
 *
 * @mixin \Drupal\KernelTests\Core\Entity\EntityKernelTestBase
 * @property \Symfony\Component\DependencyInjection\ContainerInterface $container
 */
trait TestRevisionManagerTrait {

  /**
   * Time constants for more readable test code.
   */
  protected const ONE_DAY = 86_400;
  protected const ONE_WEEK = 604_800;
  protected const ONE_MONTH = 2_592_000;

  /**
   * Configures revision manager for a specific entity type.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $plugin_id
   *   The plugin ID.
   * @param bool $enabled
   *   Whether the plugin should be enabled.
   * @param array<string, mixed> $plugin_settings
   *   The plugin configuration.
   */
  protected function configureRevisionManager(string $entity_type_id, string $plugin_id, bool $enabled, array $plugin_settings = []): void {
    $settings = $this->container->get('config.factory')->getEditable('revision_manager.settings');

    // 2) Flip the master switch.
    $enabled_entities = $settings->get('enabled_entities') ?: [];
    $enabled_entities[$entity_type_id] = $enabled;
    $settings->set('enabled_entities', $enabled_entities);

    // 3) Tweak the defaults for this plugin.
    $defaults = $settings->get('defaults') ?: [];
    $defaults[$entity_type_id][$plugin_id] = [
      'status' => $enabled,
      'settings' => $plugin_settings,
    ];

    $settings->set('defaults', $defaults)->save();

    // 4) Clear your plugin manager cache.
    $this->container->get('plugin.manager.revision_manager')->clearCachedDefinitions();
  }

  /**
   * Creates an entity with the provided values.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param array<string, mixed> $values
   *   The entity values.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   The created entity.
   */
  protected function createEntity(string $entity_type_id, array $values): object {
    $entity = $this->container->get('entity_type.manager')->getStorage($entity_type_id)->create($values);
    $entity->save();
    return $entity;
  }

  /**
   * Creates a specific number of revisions for an entity.
   *
   * @param \Drupal\Core\Entity\RevisionableInterface $entity
   *   The entity.
   * @param int $count
   *   The number of revisions to create.
   * @param array<string, mixed> $overrides
   *   Optional field overrides on each new revision (e.g. ['status' => 0]).
   */
  protected function createRevisions(RevisionableInterface $entity, int $count, array $overrides = []): void {
    $storage = $this->container->get('entity_type.manager')->getStorage($entity->getEntityTypeId());
    assert($storage instanceof RevisionableStorageInterface);

    for ($i = 0; $i < $count; $i++) {
      $revision = $storage->loadRevision($entity->getRevisionId());
      $revision->setNewRevision(TRUE);
      $revision->isDefaultRevision(FALSE);

      if ($revision instanceof FieldableEntityInterface) {
        foreach ($overrides as $field => $value) {
          $revision->set($field, $value);
        }
      }

      $revision->save();
    }
  }

  /**
   * Create a number of revisions all stamped with the given changed time.
   *
   * @param \Drupal\Core\Entity\RevisionableInterface $entity
   *   The entity to revise.
   * @param int $count
   *   How many revisions to create.
   * @param int $timestamp
   *   The UNIX timestamp to apply to each revision’s changed field.
   */
  protected function createRevisionsAtTime(RevisionableInterface $entity, int $count, int $timestamp): void {
    $storage = $this->container->get('entity_type.manager')->getStorage($entity->getEntityTypeId());
    assert($storage instanceof RevisionableStorageInterface);

    for ($i = 0; $i < $count; $i++) {
      $revision = $storage->createRevision($entity);
      if ($revision instanceof EntityChangedInterface) {
        $revision->setChangedTime($timestamp);
      }

      $revision->save();
    }
  }

  /**
   * Creates revisions before and after a cutoff date.
   *
   * @param \Drupal\Core\Entity\RevisionableInterface $entity
   *   The entity to revise.
   * @param int $older_count
   *   Number of older revisions to create (before cutoff).
   * @param int $newer_count
   *   Number of newer revisions to create (after cutoff).
   * @param int $cutoff
   *   The cutoff timestamp.
   * @param int $older_offset
   *   Time offset in seconds before the cutoff (default 1 week).
   * @param int $newer_offset
   *   Time offset in seconds after the cutoff (default 1 day).
   */
  protected function createRevisionsAroundCutoff(
    RevisionableInterface $entity,
    int $older_count,
    int $newer_count,
    int $cutoff,
    int $older_offset = self::ONE_WEEK,
    int $newer_offset = self::ONE_DAY,
  ): void {
    // Create older revisions (before cutoff)
    if ($older_count > 0) {
      $this->createRevisionsAtTime($entity, $older_count, $cutoff - $older_offset);
    }

    // Create newer revisions (after cutoff)
    if ($newer_count > 0) {
      $this->createRevisionsAtTime($entity, $newer_count, $cutoff + $newer_offset);
    }
  }

  /**
   * Process items in a queue.
   *
   * @param string $queue_id
   *   The ID of the queue to process.
   */
  protected function processRevisionQueue(string $queue_id): void {
    $queue = $this->container->get('queue')->get($queue_id);
    $worker = $this->container->get('plugin.manager.queue_worker')->createInstance($queue_id);

    assert($queue instanceof QueueInterface);
    assert($worker instanceof QueueWorkerInterface);

    while (($item = $queue->claimItem()) instanceof \stdClass) {
      $worker->processItem($item->data);
      $queue->deleteItem($item);
    }
  }

  /**
   * Asserts that an entity has the expected number of revisions.
   *
   * @param object $entity
   *   The entity.
   * @param int $expected_count
   *   The expected revision count.
   */
  protected function assertRevisionCount(object $entity, int $expected_count): void {
    $entity_type_id = $entity->getEntityTypeId();
    $entity_id = $entity->id();

    $storage = $this->entityTypeManager->getStorage($entity_type_id);
    $revision_ids = $storage->getQuery()
      ->accessCheck(FALSE)
      ->allRevisions()
      ->condition($entity->getEntityType()->getKey('id'), $entity_id)
      ->execute();

    $this->assertEquals(
      $expected_count,
      count($revision_ids),
      sprintf('Expected %d revisions for %s entity, found %d.',
        $expected_count,
        $entity_type_id,
        count($revision_ids),
      ),
    );
  }

  /**
   * Load a bundle config entity (node_type, block_content_type, etc.).
   *
   * @param string $storage_id
   *   The entity storage ID (e.g. 'node_type', 'block_content_type').
   * @param string $bundle_id
   *   The specific bundle ID (e.g. 'page', 'basic').
   *
   * @return \Drupal\Core\Config\Entity\ConfigEntityInterface|null
   *   The loaded bundle, or NULL if it doesn't exist.
   */
  protected function loadBundle(string $storage_id, string $bundle_id): ?ConfigEntityInterface {
    $entity = $this->container
      ->get('entity_type.manager')
      ->getStorage($storage_id)
      ->load($bundle_id);
    assert($entity instanceof ConfigEntityInterface);
    return $entity;
  }

}
