<?php

namespace Drupal\revision_manager;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityChangedInterface;

/**
 * Helper class for working with entities.
 */
final class EntityHelper {

  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly EntityTypeBundleInfoInterface $bundleInfo,
  ) {}

  /**
   * Gets the entity types that are supported by the revision manager.
   *
   * @return array
   *   An array of entity type IDs.
   */
  public function getSupportedEntityTypes(): array {
    $supported = [];
    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $definition) {
      if ($definition->entityClassImplements(ContentEntityInterface::class)
        && $definition->entityClassImplements(EntityChangedInterface::class)
        && $definition->isRevisionable()
        && $definition->hasLinkTemplate('canonical')
        && !$definition->isInternal()) {
        $supported[] = $entity_type_id;
      }
    }
    return $supported;
  }

  /**
   * Check if the entity type is supported or not.
   */
  public function isEntityTypeSupported(string $entity_type_id): bool {
    return in_array($entity_type_id, $this->getSupportedEntityTypes(), TRUE);
  }

  /**
   * Check if the entity type has bundles or not.
   */
  public function entityTypeHasBundles(string $entity_type_id): bool {
    $definition = $this->entityTypeManager->getDefinition($entity_type_id);
    return $definition->getBundleEntityType() !== NULL;
  }

  /**
   * Gets the bundle information for a given entity type.
   */
  public function getBundleInfo(string $entity_type_id): array {
    return $this->bundleInfo->getBundleInfo($entity_type_id) ?? [];
  }

  /**
   * Get the bundle names for a specific entity type.
   */
  public function getBundleNames(string $entity_type_id): array {
    return $this->bundleInfo->getBundleInfo($entity_type_id);
  }

  /**
   * Get the bundle key for a content entity.
   */
  public function getBundleId(ContentEntityInterface $entity): string {
    return $entity->getEntityTypeId() === 'menu_link_content'
      && method_exists($entity, 'getMenuName')
      ? $entity->getMenuName()
      : $entity->bundle();
  }

  /**
   * Get bundle‐config entity type for edge cases like menu links.
   */
  public function getBundleEntityType(string $entity_type_id): ?string {
    $bundle_type = $this->entityTypeManager
      ->getDefinition($entity_type_id)
      ->getBundleEntityType();

    if ($bundle_type !== NULL) {
      return $bundle_type;
    }
    return $entity_type_id === 'menu_link_content' ? 'menu' : NULL;
  }

  /**
   * Get the entity type for a bundle.
   */
  public function getEntityBundleOf(EntityInterface $entity): ?string {
    return $entity->getEntityTypeId() === 'menu'
      ? 'menu_link_content'
      : $entity->getEntityType()->getBundleOf();
  }

  /**
   * Gets the entity IDs by entity type and bundle.
   */
  public function getEntityIds(string $entity_type_id, ?string $bundle = NULL): array {
    if (!$this->isEntityTypeSupported($entity_type_id)) {
      return [];
    }

    $storage = $this->entityTypeManager->getStorage($entity_type_id);
    $query = $storage->getQuery()->accessCheck(FALSE);

    if ($bundle !== NULL && $this->entityTypeHasBundles($entity_type_id)) {
      $bundle_key = $entity_type_id === 'menu_link_content' ? 'menu_name' : $storage->getEntityType()->getKey('bundle');
      $query->condition($bundle_key, $bundle);
    }
    return $query->execute();
  }

}
