<?php

namespace Drupal\revision_manager\Manager;

use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\KeyValueStore\KeyValueStoreExpirableInterface;
use Drupal\revision_manager\EntityHelper;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;

/**
 * Provides functionality enqueueing entities for revision maintenance.
 */
final class QueueGenerator {
  use DependencySerializationTrait;

  /**
   * Constructs the QueueGenerator.
   */
  public function __construct(
    private readonly KeyValueStoreExpirableInterface $store,
    private readonly QueueFactory $queueFactory,
    private readonly EntityManager $entityManager,
    private readonly EntityHelper $entityHelper,
  ) {
  }

  /**
   * Enqueues an entity for revision deletion, optionally with a plugin.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $entity_id
   *   The entity ID.
   *
   * @return bool
   *   TRUE if the item was queued, FALSE otherwise.
   */
  public function enqueueEntity(string $entity_type_id, string $entity_id): bool {
    if (!$this->entityManager->isEntityTypeEnabled($entity_type_id)) {
      return FALSE;
    }
    $item = $this->buildQueueItem($entity_type_id, $entity_id);
    return $this->enqueueIfUnique($item);
  }

  /**
   * Enqueues a revision maintenance item if it is not already present.
   *
   * @param array<string, mixed> $item
   *   The item to enqueue.
   *
   * @return bool
   *   TRUE when the item was queued, FALSE when it was duplicate.
   */
  public function enqueueIfUnique(array $item): bool {
    $entity_key = $item['entity_type_id'] . $item['entity_id'];

    if ($this->store->get($entity_key)) {
      return FALSE;
    }

    // Enqueue item and set a new state that will expire in 24 hours.
    // This helps to avoid duplicate queue items.
    try {
      $this->queueFactory->get('revision_manager_remove_revisions')->createItem($item);
      $this->store->setWithExpire($entity_key, TRUE, 86400);
      return TRUE;
    }
    catch (\Exception $e) {
      return FALSE;
    }
  }

  /**
   * Enqueue specific entity ids.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param array<int, string> $entity_ids
   *   An array of entity IDs.
   */
  public function enqueueEntities(string $entity_type_id, array $entity_ids): void {
    foreach ($entity_ids as $entity_id) {
      $this->enqueueEntity($entity_type_id, $entity_id);
    }
  }

  /**
   * Enqueue all entities in a single bundle.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle
   *   The bundle name.
   */
  public function enqueueRevisionsByBundle(string $entity_type_id, string $bundle): void {
    $entity_type_ids = $this->entityHelper->getEntityIds($entity_type_id, $bundle);
    $this->enqueueEntities($entity_type_id, $entity_type_ids);
  }

  /**
   * Enqueue supported bundles of a given entity type.
   *
   * @param string $entity_type_id
   *   The entity type.
   */
  public function enqueueRevisionsByType(string $entity_type_id): void {
    foreach (array_keys($this->entityHelper->getBundleNames($entity_type_id)) as $bundle) {
      $this->enqueueRevisionsByBundle($entity_type_id, $bundle);
    }
  }

  /**
   * Builds the queue item array.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $entity_id
   *   The entity ID.
   *
   * @return array<string, mixed>
   *   The queue item array.
   */
  private function buildQueueItem(string $entity_type_id, string $entity_id): array {
    return [
      'entity_id' => $entity_id,
      'entity_type_id' => $entity_type_id,
    ];
  }

  /**
   * Empties queue and resets the duplicate‑prevention store.
   */
  public function clearQueue(): void {
    $this->queueFactory->get('revision_manager_remove_revisions')->deleteQueue();
    $this->store->deleteAll();
  }

}
