<?php

namespace Drupal\revision_manager\Plugin\QueueWorker;

use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\RevisionableStorageInterface;
use Drupal\revision_manager\Plugin\RevisionManagerPluginManager;
use Drupal\revision_manager\Manager\EntityManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\KeyValueStore\KeyValueStoreExpirableInterface;
use Drupal\Core\Entity\RevisionableInterface;

/**
 * A Queue Worker that removes revisions.
 *
 * @QueueWorker(
 *   id = "revision_manager_remove_revisions",
 *   title = @Translation("Remove entity revisions"),
 *   cron = {"time" = 120}
 * )
 */
final class RemoveRevisionsQueue extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    private readonly KeyValueStoreExpirableInterface $store,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly LoggerChannelInterface $logger,
    private readonly RevisionManagerPluginManager $pluginManager,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly EntityManager $entityManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('revision_manager.store.remove_revisions'),
      $container->get('entity_type.manager'),
      $container->get('logger.factory')->get('revision_manager'),
      $container->get('plugin.manager.revision_manager'),
      $container->get('config.factory'),
      $container->get('revision_manager.entity_manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem(mixed $data): void {
    if (!isset($data['entity_type_id']) || !isset($data['entity_id'])) {
      $this->logger->error('Missing queue data: entity_type_id or entity_id not set.');
      return;
    }

    $entity_type_id = $data['entity_type_id'];
    $entity_id = $data['entity_id'];
    $entity_key = $entity_type_id . $entity_id;

    // Try to load entity and verify it's revisionable.
    $entity_data = $this->loadEntity($entity_type_id, $entity_id);
    if (!$entity_data) {
      return;
    }

    [$entity, $storage] = $entity_data;

    // Process all enabled plugins.
    $this->processEnabledPlugins($entity_type_id, $entity, $storage);

    // Delete the processed item from the store.
    $this->store->delete($entity_key);

    // Log if verbose logging is enabled.
    if ($this->configFactory->get('revision_manager.settings')->get('verbose_log')) {
      $this->logger->info('Processed revisions for entity of type @entity_type_id with ID @entity_id.', [
        '@entity_id' => $entity_id,
        '@entity_type_id' => $entity_type_id,
      ]);
    }
  }

  /**
   * Load an entity and verify it's revisionable.
   */
  private function loadEntity(string $entity_type_id, mixed $entity_id): ?array {
    $storage = $this->entityTypeManager->getStorage($entity_type_id);
    $entity = $storage->load($entity_id);

    if (!$entity instanceof EntityInterface) {
      $this->logger->warning('Could not load entity @type:@id.', ['@type' => $entity_type_id, '@id' => $entity_id]);
      return NULL;
    }

    return [$entity, $storage];
  }

  /**
   * Process all enabled plugins.
   */
  private function processEnabledPlugins(string $entity_type_id, EntityInterface $entity, RevisionableStorageInterface $storage): void {
    if (!$entity instanceof RevisionableInterface) {
      return;
    }

    $bundle_id = $this->entityManager->getEntityBundle($entity_type_id, (string) $entity->id());
    $candidates = [];

    foreach ($this->pluginManager->getDefinitions() as $plugin_id => $plugin_definition) {
      $settings = $this->entityManager->getSettings($plugin_id, $entity_type_id, $bundle_id);
      if (isset($settings['status']) && $settings['status']) {
        $plugin = $this->pluginManager->getPlugin($plugin_id, $entity_type_id, $settings['settings']);
        $candidates[] = $plugin->deleteRevisions($entity);
      }
    }

    if ($candidates === []) {
      return;
    }

    // All enabled plugins must agree on deletions.
    $to_delete = count($candidates) === 1
      ? reset($candidates)
      : array_values(array_intersect(...$candidates));

    foreach ($to_delete as $vid) {
      $storage->deleteRevision($vid);
    }
  }

}
