<?php

namespace Drupal\revision_manager\Plugin\RevisionManager;

use Drupal\Core\Form\FormStateInterface;
use Drupal\revision_manager\Plugin\RevisionManagerBase;
use Drupal\Core\Entity\RevisionableInterface;

/**
 * Determines the number of revisions to keep based on a configurable amount.
 *
 * @RevisionManager(
 *   id = "amount",
 *   label = @Translation("Delete revisions to maintain a minimum count"),
 *   description = @Translation("Deletes revisions beyond the specified count while preserving active and forward revisions.")
 * )
 */
final class Amount extends RevisionManagerBase {

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['amount'] = [
      '#default_value' => $this->configuration['amount'] ?? 3,
      '#description' => $this->t('Deletes older revisions beyond this count. Forward revisions are always preserved.'),
      '#min' => 1,
      '#required' => TRUE,
      '#title' => $this->t('Minimum number of revisions to keep'),
      '#type' => 'number',
    ];
    return $form;
  }

  /**
   * Deletes older revisions in chunks, keeping only the newest "amount".
   */
  public function deleteRevisions(RevisionableInterface $entity): array {
    $keep = (int) ($this->configuration['amount'] ?? 3);
    $revision_key = $entity->getEntityType()->getKey('revision');
    $current = (int) $entity->getRevisionId();

    // Get all revisions older than current.
    $buckets = $this->getRevisionIds($entity, [
      ['field' => $revision_key, 'value' => $current, 'operator' => '<'],
    ]);

    // Ensures that the latest revision is kept.
    return $this->getRemovableRevisionIds($entity, $buckets, $keep);
  }

}
