<?php

namespace Drupal\Tests\revision_manager\Kernel;

use Drupal\Core\Entity\RevisionableInterface;

/**
 * Tests the Age plugin of Revision Manager for different entity types.
 *
 * @group revision_manager
 */
class AgePluginTest extends RevisionManagerTestBase {

  /**
   * Test age plugin on nodes.
   */
  public function testAgePluginNode(): void {
    $this->configureRevisionManager('node', 'age', TRUE, ['age' => 6]);
    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Node Age Test', 'status' => 1]);
    $this->assertInstanceOf(RevisionableInterface::class, $entity);

    $cutoff = (new \DateTimeImmutable())->modify('-6 months')->getTimestamp();

    // Create revisions before and after the cutoff date.
    $this->createRevisionsAroundCutoff($entity, 2, 3, $cutoff);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify the node has exactly 4 revisions remaining.
    $this->assertRevisionCount($entity, 4);
  }

  /**
   * Test age plugin on block content.
   */
  public function testAgePluginBlockContent(): void {
    $this->configureRevisionManager('block_content', 'age', TRUE, ['age' => 12]);
    $entity = $this->createEntity('block_content', ['type' => 'basic', 'info' => 'Block Age Test', 'status' => 1]);
    $this->assertInstanceOf(RevisionableInterface::class, $entity);

    $cutoff = (new \DateTimeImmutable())->modify('-12 months')->getTimestamp();

    // Create revisions with varied offsets from cutoff.
    $this->createRevisionsAroundCutoff(
      $entity, 4, 2, $cutoff,
      self::ONE_MONTH,
      self::ONE_DAY * 15,
    );
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify the block content has exactly 3 revisions remaining.
    $this->assertRevisionCount($entity, 3);
  }

  /**
   * Test age plugin on taxonomy terms.
   */
  public function testAgePluginTaxonomyTerm(): void {
    $this->configureRevisionManager('taxonomy_term', 'age', TRUE, ['age' => 1]);
    $entity = $this->createEntity('taxonomy_term', ['vid' => 'tags', 'name' => 'Term Age Test', 'status' => 1]);
    $this->assertInstanceOf(RevisionableInterface::class, $entity);

    $cutoff = (new \DateTimeImmutable())->modify('-1 month')->getTimestamp();

    // Create revisions before and after the cutoff date.
    $this->createRevisionsAroundCutoff(
      $entity, 3, 2, $cutoff,
      self::ONE_DAY * 5,
      self::ONE_DAY * 3,
    );
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify the taxonomy term has exactly 3 revisions remaining.
    $this->assertRevisionCount($entity, 3);
  }

  /**
   * Test age plugin on media entities.
   */
  public function testAgePluginMedia(): void {
    $this->configureRevisionManager('media', 'age', TRUE, ['age' => 24]);
    $entity = $this->createEntity('media', ['bundle' => 'document', 'name' => 'Media Age Test Document', 'status' => 1]);
    $this->assertInstanceOf(RevisionableInterface::class, $entity);

    $cutoff = (new \DateTimeImmutable())->modify('-24 months')->getTimestamp();

    // Create revisions before and after the cutoff date.
    $this->createRevisionsAroundCutoff(
      $entity, 5, 3, $cutoff,
      self::ONE_DAY * 60,
      self::ONE_DAY * 45,
    );
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify the media has exactly 4 revisions remaining.
    $this->assertRevisionCount($entity, 4);
  }

  /**
   * Test age plugin on group entities.
   */
  public function testAgePluginGroup(): void {
    $this->configureRevisionManager('group', 'age', TRUE, ['age' => 3]);
    $entity = $this->createEntity('group', ['type' => 'test_group_type', 'label' => 'Group Test Age Plugin']);
    $this->assertInstanceOf(RevisionableInterface::class, $entity);

    $cutoff = (new \DateTimeImmutable())->modify('-3 months')->getTimestamp();

    // Create revisions before and after the cutoff date.
    $this->createRevisionsAroundCutoff(
      $entity, 6, 3, $cutoff,
      self::ONE_MONTH,
      self::ONE_WEEK,
    );

    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify only the original + 3 newer revisions remain.
    $this->assertRevisionCount($entity, 4);
  }

  /**
   * Test age plugin on menu link content entities.
   */
  public function testAgePluginMenuLinkContent(): void {
    $this->configureRevisionManager('menu_link_content', 'age', TRUE, ['age' => 2]);
    $entity = $this->createEntity('menu_link_content', [
      'title' => 'Menu Link Age Test',
      'link' => ['uri' => 'internal:/'],
      'menu_name' => 'main',
    ]);
    $this->assertInstanceOf(RevisionableInterface::class, $entity);

    $cutoff = (new \DateTimeImmutable())->modify('-2 months')->getTimestamp();

    // Create revisions before and after the cutoff date.
    $this->createRevisionsAroundCutoff(
      $entity, 5, 2, $cutoff,
      self::ONE_WEEK * 2,
      self::ONE_DAY * 3,
    );

    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify only the original + 2 newer revisions remain.
    $this->assertRevisionCount($entity, 3);
  }

  /**
   * Test plugin disabled does nothing.
   */
  public function testAgePluginDisabled(): void {
    $this->configureRevisionManager('node', 'age', FALSE, ['age' => 3]);
    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Disabled Age Plugin Test', 'status' => 1]);
    $this->assertInstanceOf(RevisionableInterface::class, $entity);

    $this->createRevisions($entity, 5);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify the node has 6 revisions remaining since the plugin is disabled.
    $this->assertRevisionCount($entity, 6);
  }

  /**
   * Test entity type disabled prevents deletions.
   */
  public function testEntityDisabled(): void {
    $this->configureRevisionManager('node', 'age', TRUE, ['age' => 2]);

    // Disable node entity type entirely.
    $config = $this->config('revision_manager.settings');
    $enabled = $config->get('enabled_entities');
    $enabled['node'] = FALSE;
    $config->set('enabled_entities', $enabled)->save();

    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Disabled Entity Age Test', 'status' => 1]);
    $this->assertInstanceOf(RevisionableInterface::class, $entity);

    $this->createRevisions($entity, 4);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify the node has 5 revisions remaining since the entity is disabled.
    $this->assertRevisionCount($entity, 5);
  }

  /**
   * Test age plugin bundle override.
   */
  public function testAgePluginBundleOverride(): void {
    $this->configureRevisionManager('node', 'age', TRUE, ['age' => 12]);
    $bundle = $this->loadBundle('node_type', 'page');

    // Override the age plugin for the 'page' bundle to keep only 1 month.
    $bundle->setThirdPartySetting(
      'revision_manager',
      'age',
      ['status' => TRUE, 'settings' => ['age' => 1]],
    );
    $bundle->save();
    $this->container->get('plugin.manager.revision_manager')->clearCachedDefinitions();

    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Override Age Test', 'status' => 1]);
    $this->assertInstanceOf(RevisionableInterface::class, $entity);

    $cutoff = (new \DateTimeImmutable())->modify('-1 month')->getTimestamp();

    // Create revisions before and after the cutoff date.
    $this->createRevisionsAroundCutoff($entity, 2, 2, $cutoff);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify only 3 revisions remain: original + 2 newer.
    $this->assertRevisionCount($entity, 3);
  }

  /**
   * Test forward drafts (newer than current) are not deleted.
   */
  public function testForwardDraftsAreNotDeleted(): void {
    $this->configureRevisionManager('node', 'age', TRUE, ['age' => 6]);
    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Forward Drafts Age Test', 'status' => 1]);
    $this->assertInstanceOf(RevisionableInterface::class, $entity);

    $cutoff = (new \DateTimeImmutable())->modify('-6 months')->getTimestamp();

    $this->createRevisionsAtTime($entity, 3, $cutoff - self::ONE_WEEK);

    $storage = $this->container->get('entity_type.manager')->getStorage($entity->getEntityTypeId());
    $entity = $storage->load($entity->id());

    $this->createRevisions($entity, 2, ['status' => 0], FALSE);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify 1 original + 1 current + 2 forward drafts remain.
    $this->assertRevisionCount($entity, 4);
  }

  /**
   * Test no revisions are removed when all are recent.
   */
  public function testAgePluginDoesNothingWhenAllRecent(): void {
    $this->configureRevisionManager('node', 'age', TRUE, ['age' => 6]);
    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Recent Age Test', 'status' => 1]);
    $this->assertInstanceOf(RevisionableInterface::class, $entity);

    // All revisions created now (more recent than cutoff)
    $this->createRevisionsAtTime($entity, 4, time());
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify all 5 revisions remain since none are older than the cutoff.
    $this->assertRevisionCount($entity, 5);
  }

  /**
   * Test age plugin with a large number of revisions.
   */
  public function testAgePluginWithManyRevisions(): void {
    $this->configureRevisionManager('node', 'age', TRUE, ['age' => 3]);
    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Many Revisions Test', 'status' => 1]);
    $this->assertInstanceOf(RevisionableInterface::class, $entity);

    $cutoff = (new \DateTimeImmutable())->modify('-3 months')->getTimestamp();

    // Create revisions before and after the cutoff date.
    $this->createRevisionsAroundCutoff(
      $entity, 1000, 20, $cutoff,
      self::ONE_DAY * 21,
      self::ONE_DAY * 10,
    );
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify only 1 original + 20 newer.
    $this->assertRevisionCount($entity, 21);
  }

}
