<?php

/**
 * @file
 * Post update functions for Revision Manager.
 */

declare(strict_types=1);

/**
 * Update configuration values with correct types.
 */
function revision_manager_post_update_fix_config_types(&$sandbox): void {
  $fix_plugin_config = function (string $plugin_id, array $plugin_config): array {
    if (empty($plugin_config)) {
      return $plugin_config;
    }

    // Make sure the plugin ID is set.
    $plugin_config['id'] = $plugin_id;

    // The status should be a boolean.
    if (isset($plugin_config['status'])) {
      $plugin_config['status'] = (bool) $plugin_config['status'];
    }

    if (!empty($plugin_config['settings'])) {
      switch ($plugin_id) {
        case 'age':
          if ($value = $plugin_config['settings']['age'] ?? NULL) {
            $plugin_config['settings']['age'] = (int) $value;
          }
          break;

        case 'amount':
          if ($value = $plugin_config['settings']['amount'] ?? NULL) {
            $plugin_config['settings']['amount'] = (int) $value;
          }
          break;
      }
    }

    return $plugin_config;
  };

  $config_factory = \Drupal::configFactory();
  $config = $config_factory->getEditable('revision_manager.settings');

  // Fix enabled_entities types.
  $enabled_entities = $config->get('enabled_entities') ?: [];
  foreach ($enabled_entities as $entity_type_id => $enabled) {
    $enabled_entities[$entity_type_id] = (bool) $enabled;
  }
  $config->set('enabled_entities', $enabled_entities);

  // Fix defaults types.
  $defaults = $config->get('defaults') ?: [];
  foreach ($defaults as $entity_type_id => $plugins) {
    foreach ($plugins as $plugin_id => $plugin_config) {
      $defaults[$entity_type_id][$plugin_id] = $fix_plugin_config($plugin_id, $plugin_config);
    }
  }

  $config->set('defaults', $defaults)->save();

  // Fix third-party settings on bundle config entities.
  $bundle_prefixes = [
    'node.type',
    'media.type',
    'block_content.type',
    'taxonomy.vocabulary',
    'system.menu',
    'group.type',
  ];

  foreach ($bundle_prefixes as $prefix) {
    foreach ($config_factory->listAll($prefix . '.') as $config_name) {
      $bundle_config = $config_factory->getEditable($config_name);
      $third_party = $bundle_config->get('third_party_settings.revision_manager');

      if (empty($third_party)) {
        continue;
      }

      $changed = FALSE;
      foreach ($third_party as $plugin_id => $plugin_config) {
        $fixed = $fix_plugin_config($plugin_id, $plugin_config);
        if ($fixed !== $plugin_config) {
          $third_party[$plugin_id] = $fixed;
          $changed = TRUE;
        }
      }

      if ($changed) {
        $bundle_config->set('third_party_settings.revision_manager', $third_party)->save();
      }
    }
  }
}
