<?php

namespace Drupal\revision_manager\Form;

use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityFormInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Entity\EntityInterface;
use Drupal\revision_manager\RevisionManagerBatchInterface;
use Drupal\revision_manager\Manager\EntityManager;
use Drupal\revision_manager\Plugin\RevisionManagerPluginManager;

/**
 * Helper class for working with forms in Revision Manager.
 */
final class FormHelper {

  use DependencySerializationTrait;
  use StringTranslationTrait;

  /**
   * Form Helper for entity type bundle settings.
   */
  public function __construct(
    private readonly EntityManager $entityManager,
    private readonly RevisionManagerPluginManager $pluginManager,
    private readonly AccountProxyInterface $currentUser,
    private readonly RevisionManagerBatchInterface $batch,
  ) {}

  /**
   * Alters entity forms to add revision delete settings.
   *
   * @param array<string, array|string|bool> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function formAlter(array &$form, FormStateInterface $form_state): void {
    if (!$this->formAlterAccess()) {
      return;
    }

    $form_object = $form_state->getFormObject();
    if ($form_object instanceof EntityFormInterface) {
      $entity = $form_object->getEntity();
      $this->alterBundleForm($form, $form_state, $entity);
    }
  }

  /**
   * Determines whether a form can be altered.
   *
   * @return bool
   *   TRUE if a form can be altered, FALSE otherwise.
   */
  private function formAlterAccess(): bool {
    return $this->currentUser->hasPermission('administer revision_manager');
  }

  /**
   * Alters a bundle form to add revision delete settings.
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The bundle entity.
   */
  protected function alterBundleForm(array &$form, FormStateInterface $form_state, EntityInterface $entity): void {
    if (!$entity_type_id = $this->entityManager->getActiveEntityTypeIdForBundle($entity)) {
      return;
    }

    $bundle_id = $entity->id();
    $bundle_settings = $entity->isNew() ? [] : $this->entityManager->getSettingsEntityTypeBundle($entity_type_id, $bundle_id)['plugin'];

    $form['revision_manager'] = [
      '#attributes' => ['class' => ['revision-manager-settings-form']],
      '#collapsed' => FALSE,
      '#collapsible' => TRUE,
      '#group' => 'additional_settings',
      '#title' => $this->t('Revision Manager'),
      '#tree' => TRUE,
      '#type' => 'details',
      '#weight' => 10,
    ];

    $form['revision_manager'][$entity_type_id] = [
      '#tree' => TRUE,
      '#type' => 'container',
    ];

    foreach ($this->pluginManager->getDefinitions() as $plugin_id => $definition) {
      $form['revision_manager'][$entity_type_id][$plugin_id] = $this->buildPluginSubform(
        $entity_type_id,
        $plugin_id,
        $definition,
        $bundle_settings[$plugin_id] ?? [],
        $form,
        $form_state,
      );
    }

    $form['revision_manager']['enqueue_entities_now'] = $this->enqueueNowForm();

    if (isset($form['actions']['submit'])) {
      $form['actions']['submit']['#submit'][] = [$this, 'enqueueNowFormSubmit'];
    }

    $form['#attached']['library'][] = 'revision_manager/summaries';
    $form['#entity_builders'][] = [$this, 'saveBundlePluginSettings'];
  }

  /**
   * Form submission handler.
   *
   * Enqueues entities according to user setting.
   *
   * @param array<string, mixed> $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function enqueueNowFormSubmit(array &$form, FormStateInterface $form_state): void {
    if ($form_state->getValue('enqueue_entities_now')) {
      $this->batch->run();
    }
  }

  /**
   * Apply and persist Revision Manager plugin settings for a bundle.
   *
   * @param string $entity_type_id
   *   The entity type.
   * @param \Drupal\Core\Config\Entity\ConfigEntityInterface $bundle_entity
   *   The config entity type.
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function saveBundlePluginSettings(string $entity_type_id, ConfigEntityInterface $bundle_entity, array &$form, FormStateInterface $form_state): void {
    $entity_type_id = $bundle_entity->getEntityType()->getBundleOf();
    if ($entity_type_id === NULL) {
      return;
    }

    $plugin_definitions = $this->pluginManager->getDefinitions();
    foreach ($plugin_definitions as $plugin_id => $definition) {
      $plugin_subform = $form['revision_manager'][$entity_type_id][$plugin_id];
      $plugin_form_state = SubformState::createForSubform($plugin_subform, $form, $form_state);

      // Get the bundle settings.
      $bundle_settings = $plugin_form_state->getValues();
      $bundle_settings['id'] = $plugin_id;

      // Ensure the status is stored as a boolean.
      $bundle_settings['status'] = !empty($bundle_settings['status']);

      // Instantiate the plugin with the submitted settings, so that it has a
      // chance to process its configuration.
      $plugin = $this->pluginManager->getPlugin($plugin_id, $entity_type_id, $bundle_settings['settings'] ?? []);
      $bundle_settings['settings'] = $plugin->getConfiguration();

      $default_settings = $this->pluginManager->getDefaultPluginSettings($plugin_id, $entity_type_id);

      // If the settings are different from the default, save them.
      if ($bundle_settings !== $default_settings) {
        $bundle_entity->setThirdPartySetting('revision_manager', $plugin_id, $bundle_settings);
      }
      else {
        $bundle_entity->unsetThirdPartySetting('revision_manager', $plugin_id);
      }
    }
  }

  /**
   * Adds the 'enqueue all entities for revision deletion' checkbox to the form.
   *
   * @return array
   *   The checkbox form element definition.
   */
  public function enqueueNowForm(): array {
    return [
      '#type' => 'checkbox',
      '#parents' => ['enqueue_entities_now'],
      '#title' => $this->t('Enqueue all enabled entities for revision deletion after hitting <em>Save</em>'),
      '#description' => $this->t('This setting will enqueue all enabled entities including the above changes for all enabled plugins.'),
      '#access' => $this->currentUser->hasPermission('administer revision_manager'),
      '#default_value' => FALSE,
      '#weight' => 90,
    ];
  }

  /**
   * Builds the plugin sub-form.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $plugin_id
   *   The plugin ID.
   * @param array<string, mixed> $plugin_definition
   *   The plugin definition.
   * @param array<string, mixed> $settings
   *   The default settings for the plugin.
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   Render array for the plugin container.
   */
  public function buildPluginSubform(
    string $entity_type_id,
    string $plugin_id,
    array $plugin_definition,
    array $settings,
    array $form,
    FormStateInterface $form_state,
  ): array {
    $plugin = $this->pluginManager->getPlugin(
      $plugin_id,
      $entity_type_id,
      $settings['settings'] ?? [],
    );

    $element = [
      '#type' => 'details',
      '#title' => $plugin_definition['label'],
      '#open' => $settings['status'] ?? 0,
      '#tree' => TRUE,
      '#attributes' => ['class' => ['revision-manager-plugin-settings']],
      'status' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Enabled'),
        '#default_value' => $settings['status'] ?? 0,
      ],
      'settings' => [
        '#type' => 'fieldset',
        '#states' => [
          'visible' => [
            sprintf(
              ':input[name="%s[%s][status]"]',
              $entity_type_id,
              $plugin_id,
            ) => ['checked' => TRUE],
          ],
        ],
      ],
    ];

    $sub_state = SubformState::createForSubform($element['settings'], $form, $form_state);
    $element['settings'] = $plugin->buildConfigurationForm($element['settings'], $sub_state);

    return $element;
  }

}
