<?php

namespace Drupal\revision_manager\Manager;

use Drupal\Core\Entity\EntityInterface;
use Drupal\revision_manager\EntityHelper;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\revision_manager\Plugin\RevisionManagerPluginManager;

/**
 * Provides entity management functionality for the Revision Manager module.
 */
final class EntityManager {

  /**
   * Cache of merged settings from config and node-type third-party settings.
   */
  protected array $settings = [];

  public function __construct(
    private readonly ConfigFactoryInterface $configFactory,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly EntityHelper $entityHelper,
    private readonly RevisionManagerPluginManager $pluginManager,
  ) {}

  /**
   * Check the given entity type is globally enabled for Revision Manager.
   */
  public function isEntityTypeEnabled(string $entity_type_id): bool {
    $config = $this->configFactory->get('revision_manager.settings');
    $enabled_entities = $config->get('enabled_entities') ?? [];
    return (bool) ($enabled_entities[$entity_type_id] ?? FALSE);
  }

  /**
   * Checks if the given entity bundle has enabled plugins.
   */
  public function hasEnabledPluginsForEntityBundle(string $entity_type_id, string $entity_id): bool {
    $bundle_id = $this->getEntityBundle($entity_type_id, $entity_id);
    if ($bundle_id === NULL) {
      return FALSE;
    }

    $plugins = $this->getSettingsEntityTypeBundle($entity_type_id, $bundle_id)['plugin'] ?? [];
    $enabled_plugins = array_filter($plugins, static fn ($plugin) => (bool) ($plugin['status'] ?? FALSE));
    return $enabled_plugins !== [];
  }

  /**
   * Check if the entity type is enabled and the bundle has enabled plugins.
   */
  public function isEntityEnabledAndHasEnabledPlugins(string $entity_type_id, string $entity_id): bool {
    return $this->isEntityTypeEnabled($entity_type_id)
      && $this->hasEnabledPluginsForEntityBundle($entity_type_id, $entity_id);
  }

  /**
   * Returns the entity type ID if Revision Manager is enabled for it.
   *
   * @param \Drupal\Core\Entity\EntityInterface $bundle_entity
   *   A bundle config entity (e.g., node_type, media_type).
   *
   * @return string|null
   *   The entity type ID (e.g., 'node', 'media') when enabled,
   *   or NULL if the bundle is not a content bundle or is disabled.
   */
  public function getActiveEntityTypeIdForBundle(EntityInterface $bundle_entity): ?string {
    $entity_type_id = $this->entityHelper->getEntityBundleOf($bundle_entity);
    return $entity_type_id && $this->isEntityTypeEnabled($entity_type_id) ? $entity_type_id : NULL;
  }

  /**
   * Load an entity and return its bundle, or NULL if it doesn't exist.
   */
  public function getEntityBundle(string $entity_type_id, string $entity_id): ?string {
    $entity = $this->entityTypeManager
      ->getStorage($entity_type_id)
      ->load($entity_id);

    return $entity instanceof ContentEntityInterface
      ? $this->entityHelper->getBundleId($entity)
      : NULL;
  }

  /**
   * Retrieves merged settings for all entity types and their bundles.
   */
  private function getAllSettings(): array {
    if ($this->settings) {
      return $this->settings;
    }

    foreach ($this->entityHelper->getSupportedEntityTypes() as $entity_type_id) {
      $bundle_entity_type = $this->entityHelper->getBundleEntityType($entity_type_id);

      // Skip truly bundle-less entities (no bundle entity type at all).
      if (!$bundle_entity_type) {
        continue;
      }

      // Handle bundleless entities with pseudo-bundle pattern.
      if (!$this->entityHelper->hasBundles($entity_type_id)) {
        $bundle_plugin_defaults = [];
        foreach (array_keys($this->pluginManager->getDefinitions()) as $plugin_id) {
          $bundle_plugin_defaults[$plugin_id] =
            $this->pluginManager->getDefaultPluginSettings($plugin_id, $entity_type_id);
        }
        $this->settings[$entity_type_id][$entity_type_id] = [
          'plugin' => $bundle_plugin_defaults,
          'status' => 'default',
        ];
        continue;
      }

      $bundles = $this->entityTypeManager->getStorage($bundle_entity_type)->loadMultiple();
      foreach ($bundles as $bundle) {
        // Skip if the bundle is not a config entity with third-party settings.
        if (!$bundle instanceof ConfigEntityInterface) {
          continue;
        }

        $bundle_id = $bundle->id();
        $bundle_plugin_defaults = $defaults[$entity_type_id][$bundle_id] ??= [];

        // Build default settings for each plugin.
        foreach (array_keys($this->pluginManager->getDefinitions()) as $plugin_id) {
          $bundle_plugin_defaults[$plugin_id] =
            $this->pluginManager->getDefaultPluginSettings($plugin_id, $entity_type_id);
        }

        // Get bundle-specific overrides.
        $bundle_settings = $bundle->getThirdPartySettings('revision_manager');
        $bundle_settings_overrides = array_intersect_key($bundle_settings, $bundle_plugin_defaults);

        // Merge and store settings.
        $this->settings[$entity_type_id][$bundle_id] = [
          'plugin' => array_merge($bundle_plugin_defaults, $bundle_settings_overrides),
          'status' => $bundle_settings_overrides === [] ? 'default' : 'overridden',
        ];
      }
    }

    return $this->settings;
  }

  /**
   * Gets the plugin-specific settings for a given entity type + bundle.
   */
  public function getSettings(string $plugin_id, string $entity_type_id, string $bundle_id): array {
    $settings = $this->getAllSettings();
    return $settings[$entity_type_id][$bundle_id]['plugin'][$plugin_id] ?? [];
  }

  /**
   * Gets the entire block of settings for a given entity type + bundle.
   */
  public function getSettingsEntityTypeBundle(string $entity_type_id, string $bundle_id): array {
    $settings = $this->getAllSettings();
    return $settings[$entity_type_id][$bundle_id] ?? [];
  }

  /**
   * Reset any cached settings in memory.
   */
  public function resetSettingsCache(): void {
    $this->settings = [];
  }

}
