<?php

namespace Drupal\revision_manager;

/**
 * An interface for batch controller to process entities for revision cleanup.
 *
 * @package Drupal\revision_manager
 */
interface RevisionManagerBatchInterface {

  /**
   * Default number of entities to process in each batch operation.
   *
   * Higher value is efficient as this only queues items, not processes them.
   * Actual revision cleanup occurs later when queue workers run.
   */
  public const BATCH_SIZE = 2500;

  public const ENQUEUE_TYPE_FORM = 'form';
  public const ENQUEUE_TYPE_DRUSH = 'drush';

  /**
   * Runs the batch process to enqueue entities for revision cleanup.
   *
   * @param string $from
   *   Use a normal Drupal batch ('form') or immediately via Drush ('drush').
   *
   * @return bool
   *   FALSE if there were no entities to enqueue; TRUE otherwise.
   */
  public function run(string $from = self::ENQUEUE_TYPE_FORM): bool;

  /**
   * Initializes the batch.
   *
   * @param int $total_items
   *   The total number of items to be processed in the batch.
   * @param array<string,mixed> $context
   *   Batch context array, which stores information about the batch process.
   */
  public function initBatch(int $total_items, array &$context): void;

  /**
   * Processes a chunk of entities for revision cleanup.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param array<int,mixed> $entity_ids
   *   An array of entity IDs to process.
   * @param array<string,mixed> $context
   *   The batch context array.
   */
  public function processEntityChunk(string $entity_type_id, array $entity_ids, array &$context): void;

  /**
   * Callback function called by the batch API when all operations are finished.
   *
   * @param bool $success
   *   Indicates whether the batch process was successful.
   * @param array<string,mixed> $results
   *   Results information passed from the processing callback.
   * @param array<int,array{0:callable,1:array<mixed>}> $operations
   *   A list of the operations that had not been completed by the batch API.
   *
   * @return bool
   *   Indicates whether the batch process was successful.
   */
  public function finishBatch(bool $success, array $results, array $operations): bool;

}
