<?php

namespace Drupal\Tests\revision_manager\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Test the Revision manager configuration settings form.
 *
 * @group revision_manager
 */
class RevisionManagerConfigFormTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'revision_manager',
    'node',
    'taxonomy',
    'block_content',
    'media',
    'menu_ui',
  ];

  /**
   * Test backend Revision Manager configuration settings form.
   */
  public function testRevisionManagerOverview(): void {
    // The list of entity types.
    $entity_types = [
      'node',
      'media',
      'taxonomy_term',
      'block_content',
    ];

    // Log in as an admin user to test admin pages.
    $permissions = [
      'administer revision_manager',
    ];
    $adminUser = $this->drupalCreateUser($permissions);
    $this->drupalLogin($adminUser);

    $this->drupalGet('/admin/config/content/revision-manager');

    $this->assertSession()->statusCodeEquals(200);

    $this->assertSession()->pageTextContains('Revision Manager');

    foreach ($entity_types as $entity_type) {
      $entity_type = str_replace('_', '-', $entity_type);

      // Verify each entity type has its accordion.
      $this->assertSession()->elementExists('css', "form#revision-manager-settings details#edit-{$entity_type}");
      // Verify each entity type has “Enabled” checkbox and
      // the plugin fieldsets underneath.
      $this->assertSession()->elementExists('css', "form#revision-manager-settings details input#edit-{$entity_type}-enable-entity-type");
      $this->assertSession()->elementExists('css', "form#revision-manager-settings details details#edit-{$entity_type}-settings-amount");
      $this->assertSession()->elementExists('css', "form#revision-manager-settings details details#edit-{$entity_type}-settings-age");
      $this->assertSession()->elementExists('css', "form#revision-manager-settings details table#edit-{$entity_type}-settings-table");
    }
  }

  /**
   * Test access and UI visibility without proper permission.
   */
  public function testPermissionChecks(): void {
    // Log in as a user without administer revision_manager permission.
    $permissions = [
      'access content',
    ];
    $basicUser = $this->drupalCreateUser($permissions);
    $this->drupalLogin($basicUser);

    $this->drupalGet('/admin/config/content/revision-manager');
    $this->assertSession()->statusCodeEquals(403);

    // Visit a bundle edit form (e.g., page content type).
    $this->drupalGet('/admin/structure/types/manage/page');
    // Confirm the "Revision Manager" vertical tab or section is NOT present.
    $this->assertSession()->pageTextNotContains('Revision Manager');
  }

  /**
   * Test toggling the watchdog and auto-queue checkboxes.
   */
  public function testWatchdogAndAutoQueueSettings(): void {
    // Create and login an admin user.
    $adminUser = $this->drupalCreateUser(['administer revision_manager']);
    $this->drupalLogin($adminUser);

    // Visit the settings form.
    $this->drupalGet('/admin/config/content/revision-manager');

    // Flip the settings: enable both checkboxes.
    $edit = [
      'verbose_log' => TRUE,
      'disable_automatic_queueing' => TRUE,
    ];
    $this->submitForm($edit, 'Save configuration');

    // Assert the success message appears.
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Assert that config was saved correctly.
    $config = $this->config('revision_manager.settings');
    $this->assertTrue($config->get('verbose_log'), 'Log to watchdog setting was saved.');
    $this->assertTrue($config->get('disable_automatic_queueing'), 'Disable auto-queue setting was saved.');
  }

}
