<?php

declare(strict_types=1);

namespace Drupal\Tests\revision_manager\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\block_content\Entity\BlockContentType;
use Drupal\node\Entity\Node;
use Drupal\media\Entity\Media;
use Drupal\taxonomy\Entity\Term;
use Drupal\block_content\Entity\BlockContent;
use Drupal\media\Entity\MediaType;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\field\Entity\FieldConfig;
use Drupal\Tests\node\Traits\ContentTypeCreationTrait;
use Drupal\taxonomy\Entity\Vocabulary;

/**
 * Tests the Bundle configure link functionality.
 *
 * @group revision_manager
 */
class RevisionManagerBundleConfigureLinkTest extends WebDriverTestBase {

  use ContentTypeCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'revision_manager',
    'node',
    'taxonomy',
    'block_content',
    'media',
    'menu_ui',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'olivero';

  protected const REVISION_MANAGER_PATH = '/admin/config/content/revision-manager';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->drupalLogin($this->drupalCreateUser([
      'administer revision_manager',
      'administer content types',
      'administer site configuration',
      'administer media types',
      'administer taxonomy',
      'administer blocks',
      'administer block types',
      'administer block content',
      'administer menu',
    ]));

    // Create Article and Basic page content types.
    foreach (['article' => 'Article', 'page' => 'Basic page'] as $machine_name => $label) {
      $this->createContentType([
        'type' => $machine_name,
        'name' => $label,
        'new_revision' => TRUE,
      ])->save();
    }

    $this->createMediaTypeWithImageField('test_image', 'Test Image');

    // Create a custom block type if it doesn't exist.
    if (!BlockContentType::load('basic_block')) {
      BlockContentType::create([
        'id' => 'basic_block',
        'label' => 'Basic Block',
        'revision' => TRUE,
      ])->save();
    }

    // Create test vocabulary.
    if (!Vocabulary::load('test')) {
      Vocabulary::create([
        'vid' => 'test',
        'description' => 'Test vocabulary',
        'name' => 'Test',
      ])->save();
    }

    // Important to refresh system state.
    $this->resetAll();
  }

  /**
   * Test Revision Manager Enqueue now.
   */
  public function testRevisionManagerEnqueue(): void {
    // Empty queue.
    $queue = \Drupal::service('queue')->get('revision_manager_remove_revisions');
    $queue->deleteQueue();
    // Create test data.
    Node::create(['type' => 'page', 'title' => "Test Page 1"])->save();
    Media::create(['bundle' => 'test_image', 'name' => 'Test Media 1'])->save();
    Term::create(['vid' => 'test', 'name' => 'Test taxonomy 1'])->save();
    BlockContent::create(['type' => 'basic_block', 'info' => 'Test Block 1'])->save();
    // Go to revision manager config.
    $this->drupalGet(self::REVISION_MANAGER_PATH);

    // Loop over supported entity types.
    $entity_types = [
      'node' => 1,
      'media' => 1,
      'taxonomy_term' => 1,
      'block_content' => 1,
    ];

    foreach ($entity_types as $entity_type => $count) {
      $entity_type_name = str_replace('_', '-', $entity_type);
      $details = $this->getSession()->getPage()->find('css', "form#revision-manager-settings #edit-{$entity_type_name}");
      $summary = $details->find('xpath', './summary');
      $summary->click();
      $edit["{$entity_type}[enable_entity_type]"] = TRUE;
    }
    $edit["enqueue_entities_now"] = TRUE;
    $this->submitForm($edit, 'edit-submit');
    $this->assertSession()->waitForText('The configuration options have been saved.', 5000);
    // Check queue contents.
    $queue = \Drupal::service('queue')->get('revision_manager_remove_revisions');
    $items = [];
    while ($item = $queue->claimItem()) {
      $items[] = $item->data;
      $queue->deleteItem($item);
    }

    $expected_count = array_sum($entity_types);
    $this->assertCount($expected_count, $items, "Queue contains $expected_count items total.");

    // Optionally: group by entity type to validate distribution.
    $by_type = [];
    foreach ($items as $item) {
      $by_type[$item['entity_type_id']][] = $item;
    }

    foreach ($entity_types as $type => $count) {
      $actual = count($by_type[$type] ?? []);
      $this->assertEquals($count, $actual, "Queue contains $count items for $type.");
    }
  }

  /**
   * Tests Show Hide plugins.
   */
  public function testShowHidePlugins(): void {

    // The list of entity types.
    $entity_types = [
      'node',
      'media',
      'taxonomy_term',
      'block_content',
    ];

    $this->drupalGet(self::REVISION_MANAGER_PATH);
    $page = $this->getSession()->getPage();
    foreach ($entity_types as $entity_type) {
      $entity_type = str_replace('_', '-', $entity_type);
      $details = $this->getSession()->getPage()->find('css', "form#revision-manager-settings #edit-{$entity_type}");
      $summary = $details->find('xpath', './summary');
      $this->assertNotNull($summary, 'Summary tag found in details.');
      $summary->click();

      // Confirm the checkbox is present.
      $checkbox = $page->findField("edit-{$entity_type}-enable-entity-type");
      $this->assertNotNull($checkbox, "{$entity_type} Enabled checkbox found.");

      // Check the plugin.
      $checkbox->check();

      // Wait for Amount and Age plugin forms to appear.
      $this->assertSession()->waitForElementVisible('css', "#edit-{$entity_type}-settings-amount");
      $this->assertSession()->waitForElementVisible('css', "#edit-{$entity_type}-settings-age");

      // Wait for the bundle table to appear.
      $this->assertSession()->waitForElementVisible('css', "#edit-{$entity_type}-settings-table");

      // Uncheck the plugin.
      $checkbox->uncheck();

      // Wait for all of them to disappear again.
      // Check if the element exists and wait for it to become invisible.
      $amountElement = $page->find('css', "#edit-{$entity_type}-settings-amount");
      $this->assertNotNull($amountElement, 'Amount Element found.');
      $this->assertFalse($amountElement->isVisible(), 'Amount Element is not visible.');
      $ageElement = $page->find('css', "#edit-{$entity_type}-settings-age");
      $this->assertNotNull($ageElement, 'Age Element found.');
      $this->assertFalse($ageElement->isVisible(), 'Age Element is not visible.');
      $tableElement = $page->find('css', "#edit-{$entity_type}-settings-table");
      $this->assertNotNull($tableElement, 'Table Element found.');
      $this->assertFalse($tableElement->isVisible(), 'Table Element is not visible.');
    }
  }

  /**
   * Node: Test Configure link navigation for Basic page bundle.
   */
  public function testConfigurePageBundleLink(): void {

    // Go to revision manager config.
    $this->drupalGet(self::REVISION_MANAGER_PATH);

    $page = $this->getSession()->getPage();

    // Check default values.
    $config = $this->config('revision_manager.settings');
    $default_enabled = $config->get('third_party_settings.revision_manager');
    $this->assertEquals(0, $default_enabled, 'Default plugin is disabled by default.');

    // Expand the "Content" <details> element.
    $this->assertSession()->waitForElementVisible('css', '#edit-node summary', 5000);
    $page->find('css', '#edit-node summary')->click();

    $edit = [
      'node[enable_entity_type]' => TRUE,
    ];
    $this->submitForm($edit, 'Save configuration');

    // Assert the success message appears.
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Wait for the settings table.
    $this->assertSession()->waitForElementVisible('css', 'table#edit-node-settings-table', 5000);

    // Find the Configure link for the "page" bundle row.
    $links = $page->findAll('css', 'table#edit-node-settings-table a');

    foreach ($links as $link) {
      $url = $link->getAttribute('href');
      if (str_contains($url, 'page')) {
        $this->drupalGet($url);
        $links = $this->getSession()->getPage()->findAll('css', '.vertical-tabs__menu a');
        $found = FALSE;
        foreach ($links as $link) {
          $title_element = $link->find('css', '.vertical-tabs__menu-item-title');
          if ($title_element && trim($title_element->getText()) === 'Revision Manager') {
            $found = TRUE;
            // Confirm Revision Manager tab is visible.
            $this->assertTrue($link->isVisible(), 'Revision Manager tab is visible.');

            // Optionally, confirm it has correct href.
            $this->assertEquals('#edit-revision-manager', $link->getAttribute('href'));
            break;
          }
        }

        $this->assertTrue($found, 'Revision Manager tab was found in the tab list.');
      }
    }

    // Bundle overrides stick.
    // Go to Revision Manager tab.
    $page = $this->getSession()->getPage();
    $this->assertSession()->waitForElementVisible('css', '.vertical-tabs__menu a', 5000);
    $tab = $page->findLink('Revision Manager');
    $this->assertNotNull($tab, 'Revision Manager tab found.');
    $tab->click();
    $this->assertSession()->waitForElementVisible('css', '#edit-revision-manager', 5000);

    // Enable a plugin checkbox
    // (e.g., Delete revisions to maintain a minimum count).
    $this->assertSession()->waitForElementVisible('css', '#edit-revision-manager-node-amount-status', 5000);
    // Expand the plugin <details> accordion first.
    $page->find('css', 'details#edit-revision-manager-node-amount summary')->click();
    $this->assertSession()->waitForElementVisible('css', '#edit-revision-manager-node-amount-status', 5000);

    // Then check the box.
    $edit = [
      'revision_manager[node][amount][status]' => TRUE,
    ];
    $this->submitForm($edit, 'edit-submit');
    $this->assertSession()->pageTextContains('The content type Basic page has been updated.');

    // Check config: node_type.page should now have third-party settings.
    $config = \Drupal::configFactory()->get('node.type.page')->get('third_party_settings.revision_manager.amount.status');
    $this->assertEquals(1, $config);

    // Go back to Revision Manager overview and verify "Overridden".
    $this->drupalGet(self::REVISION_MANAGER_PATH);
    $this->assertSession()->waitForElementVisible('css', '#edit-node summary', 5000);
    $page->find('css', '#edit-node summary')->click();
    $this->assertSession()->waitForElementVisible('css', 'table#edit-node-settings-table', 5000);

    $row = $page->find('xpath', '//table[@id="edit-node-settings-table"]//tr[td[2][contains(text(), "page")]]');
    $this->assertNotNull($row, 'Page bundle row found.');
    $this->assertTrue(strpos($row->getText(), 'Overridden') !== FALSE, 'Page bundle shows "Overridden".');

    // Override a bundle, then revert to defaults (uncheck “Enabled”) and save;
    // Confirm the third-party setting is removed.
    $this->drupalGet('/admin/structure/types/manage/page');
    $page = $this->getSession()->getPage();
    $this->assertSession()->waitForElementVisible('css', '.vertical-tabs__menu a', 5000);
    $tab = $page->findLink('Revision Manager');
    $this->assertNotNull($tab, 'Revision Manager tab found.');
    $tab->click();
    $this->assertSession()->waitForElementVisible('css', '#edit-revision-manager', 5000);
    $edit = ['revision_manager[node][amount][status]' => FALSE];
    $this->submitForm($edit, 'edit-submit');
    $this->assertSession()->pageTextContains('The content type Basic page has been updated.');
    $config = \Drupal::config('node.type.page')->get('third_party_settings.revision_manager.amount.status');
    $this->assertEmpty($config, 'Third-party setting is removed.');
  }

  /**
   * Media: Test Revision Manager plugin toggling creates/removes overrides.
   */
  public function testRevisionManagerPluginOverrideForMediaType(): void {

    // Go to revision manager config.
    $this->drupalGet(self::REVISION_MANAGER_PATH);

    $page = $this->getSession()->getPage();

    // Expand the "Content" <details> element.
    $this->assertSession()->waitForElementVisible('css', '#edit-media summary', 5000);
    $page->find('css', '#edit-media summary')->click();

    $edit = [
      'media[enable_entity_type]' => TRUE,
    ];
    $this->submitForm($edit, 'Save configuration');

    // Assert the success message appears.
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Wait for the settings table.
    $this->assertSession()->waitForElementVisible('css', 'table#edit-media-settings-table', 5000);

    // Visit media type edit form.
    $this->drupalGet('/admin/structure/media/manage/test_image');

    $page = $this->getSession()->getPage();

    $links = $this->getSession()->getPage()->findAll('css', '.vertical-tabs__menu a');
    $found = FALSE;
    foreach ($links as $link) {
      $title_element = $link->find('css', '.vertical-tabs__menu-item-title');
      if ($title_element && trim($title_element->getText()) === 'Revision Manager') {
        $found = TRUE;
        // Confirm it's visible.
        $this->assertTrue($link->isVisible(), 'Revision Manager tab is visible.');

        // Optionally, confirm it has correct href.
        $this->assertEquals('#edit-revision-manager', $link->getAttribute('href'));
        break;
      }
    }

    $this->assertTrue($found, 'Revision Manager tab was found in the tab list.');

    $this->assertSession()->waitForElementVisible('css', '.vertical-tabs__menu a', 5000);
    $tab = $page->findLink('Revision Manager');
    $this->assertNotNull($tab, 'Revision Manager tab found.');
    $tab->click();
    $this->assertSession()->waitForElementVisible('css', '#edit-revision-manager', 5000);

    // Enable a plugin checkbox
    // (e.g., Delete revisions to maintain a minimum count).
    $this->assertSession()->waitForElementVisible('css', '#edit-revision-manager-media-amount-status', 5000);
    // Expand the plugin <details> accordion first.
    $page->find('css', 'details#edit-revision-manager-media-amount summary')->click();
    $this->assertSession()->waitForElementVisible('css', '#edit-revision-manager-media-amount-status', 5000);

    // Save.
    $edit = [
      'revision_manager[media][amount][status]' => TRUE,
    ];
    $this->submitForm($edit, 'edit-submit');
    $this->assertSession()->pageTextContains('The media type Test Image has been updated.');

    // Check third-party config now exists.
    $config = \Drupal::config('media.type.test_image')->get('third_party_settings.revision_manager.amount.status');
    $this->assertEquals(1, $config, 'Plugin override saved.');

    // Now disable it.
    $this->drupalGet('/admin/structure/media/manage/test_image');
    $page = $this->getSession()->getPage();
    $this->assertSession()->waitForElementVisible('css', '.vertical-tabs__menu a', 5000);
    $tab = $page->findLink('Revision Manager');
    $this->assertNotNull($tab, 'Revision Manager tab found.');
    $tab->click();
    $this->assertSession()->waitForElementVisible('css', '#edit-revision-manager', 5000);
    $edit = [
      'revision_manager[media][amount][status]' => FALSE,
    ];
    $this->submitForm($edit, 'edit-submit');
    $this->assertSession()->pageTextContains('The media type Test Image has been updated.');

    $config = \Drupal::config('media.type.test_image')->get('third_party_settings.revision_manager.amount.status');
    $this->assertEmpty($config, 'Plugin override removed.');
  }

  /**
   * Block type: Test Revision Manager.
   *
   * Plugin toggling creates/removes overrides.
   */
  public function testRevisionManagerPluginOverrideForBlockType(): void {

    // Go to revision manager config.
    $this->drupalGet(self::REVISION_MANAGER_PATH);

    $page = $this->getSession()->getPage();

    // Expand the "Content" <details> element.
    $this->assertSession()->waitForElementVisible('css', '#edit-block-content summary', 5000);
    $page->find('css', '#edit-block-content summary')->click();

    $edit = [
      'block_content[enable_entity_type]' => TRUE,
    ];
    $this->submitForm($edit, 'Save configuration');

    // Assert the success message appears.
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Wait for the settings table.
    $this->assertSession()->waitForElementVisible('css', 'table#edit-block-content-settings-table', 5000);

    // Visit media type edit form.
    $this->drupalGet('/admin/structure/block-content/manage/basic_block');

    $page = $this->getSession()->getPage();

    $this->assertSession()->elementExists('css', "details#edit-revision-manager");
    $this->assertSession()->pageTextContains('Revision Manager');

    // Enable a plugin checkbox
    // (e.g., Delete revisions to maintain a minimum count).
    $details = $page->find('css', 'details#edit-revision-manager');
    $summary = $details->find('xpath', './summary');
    $summary->click();
    $page->find('css', 'details#edit-revision-manager-block-content-amount summary')->click();
    $this->assertSession()->waitForElementVisible('css', '#edit-revision-manager-block-content-amount-status', 5000);

    // Save.
    $edit = ['revision_manager[block_content][amount][status]' => TRUE];
    $this->submitForm($edit, 'edit-submit');
    $this->assertSession()->pageTextContains('Block type Basic block has been updated.');

    // Check third-party config now exists.
    $config = \Drupal::config('block_content.type.basic_block')->get('third_party_settings.revision_manager.amount.status');
    $this->assertEquals(1, $config, 'Plugin override saved.');

    // Now disable it.
    $this->drupalGet('/admin/structure/block-content/manage/basic_block');
    $page = $this->getSession()->getPage();
    $this->assertSession()->elementExists('css', "details#edit-revision-manager");
    $this->assertSession()->pageTextContains('Revision Manager');

    $details = $page->find('css', 'details#edit-revision-manager');
    $summary = $details->find('xpath', './summary');
    $summary->click();
    $this->assertSession()->waitForElementVisible('css', '#edit-revision-manager-block-content-amount-status', 5000);
    $edit = [
      'revision_manager[block_content][amount][status]' => FALSE,
    ];
    $this->submitForm($edit, 'edit-submit');
    $this->assertSession()->pageTextContains('Block type Basic block has been updated.');

    $config = \Drupal::config('block_content.type.basic_block')->get('third_party_settings.revision_manager.amount.status');
    $this->assertEmpty($config, 'Plugin override removed.');
  }

  /**
   * Taxonomy: Test Revision Manager plugin toggling creates/removes overrides.
   */
  public function testRevisionManagerPluginOverrideForTaxonomyType(): void {

    // Go to revision manager config.
    $this->drupalGet(self::REVISION_MANAGER_PATH);

    $page = $this->getSession()->getPage();

    // Expand the "Content" <details> element.
    $details = $page->find('css', 'details#edit-taxonomy-term');
    $summary = $details->find('xpath', './summary');
    $summary->click();

    $edit = [
      'taxonomy_term[enable_entity_type]' => TRUE,
    ];
    $this->submitForm($edit, 'Save configuration');

    // Assert the success message appears.
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Wait for the settings table.
    $this->assertSession()->waitForElementVisible('css', 'table#edit-taxonomy-term-settings-table', 5000);

    // Visit media type edit form.
    $this->drupalGet('/admin/structure/taxonomy/manage/test');

    $page = $this->getSession()->getPage();

    $this->assertSession()->elementExists('css', "details#edit-revision-manager");
    $this->assertSession()->pageTextContains('Revision Manager');

    // Enable a plugin checkbox
    // (e.g., Delete revisions to maintain a minimum count).
    $details = $page->find('css', 'details#edit-revision-manager');
    $summary = $details->find('xpath', './summary');
    $summary->click();
    $page->find('css', 'details#edit-revision-manager-taxonomy-term-amount summary')->click();
    $this->assertSession()->waitForElementVisible('css', '#edit-revision-manager-taxonomy-term-amount-status', 5000);

    // Save.
    $edit = [
      'revision_manager[taxonomy_term][amount][status]' => TRUE,
    ];
    $this->submitForm($edit, 'edit-submit');
    $this->assertSession()->pageTextContains('Updated vocabulary Test.');

    // Check third-party config now exists.
    $config = \Drupal::config('taxonomy.vocabulary.test')->get('third_party_settings.revision_manager.amount.status');
    $this->assertEquals(1, $config, 'Plugin override saved.');

    // Now disable it.
    $this->drupalGet('/admin/structure/taxonomy/manage/test');
    $page = $this->getSession()->getPage();
    $this->assertSession()->elementExists('css', "details#edit-revision-manager");
    $this->assertSession()->pageTextContains('Revision Manager');

    // $page->find('css', '.revision-manager-settings-form summary')->click();
    $details = $page->find('css', 'details#edit-revision-manager');
    $summary = $details->find('xpath', './summary');
    $summary->click();
    $this->assertSession()->waitForElementVisible('css', '#edit-revision-manager-taxonomy-term-amount-status', 5000);
    $edit = [
      'revision_manager[taxonomy_term][amount][status]' => FALSE,
    ];
    $this->submitForm($edit, 'edit-submit');
    $this->assertSession()->pageTextContains('Updated vocabulary Test.');

    $config = \Drupal::config('taxonomy.vocabulary.test')->get('third_party_settings.revision_manager.amount.status');
    $this->assertEmpty($config, 'Plugin override removed.');
  }

  /**
   * Test Revision Manager form validations.
   */
  public function testFormValidate(): void {
    // Go to revision manager config.
    $this->drupalGet(self::REVISION_MANAGER_PATH);

    // Loop over supported entity types.
    $entity_types = [
      'node' => 1,
      'media' => 1,
      'taxonomy_term' => 1,
      'block_content' => 1,
    ];

    // Check form validations on revision manager configuration form.
    $page = $this->getSession()->getPage();
    foreach ($entity_types as $entity_type => $count) {
      $entity_type_name = str_replace('_', '-', $entity_type);
      $details = $this->getSession()->getPage()->find('css', "form#revision-manager-settings #edit-{$entity_type_name}");
      $summary = $details->find('xpath', './summary');
      $summary->click();
      $checkbox = $page->findField("edit-{$entity_type_name}-enable-entity-type");
      $checkbox->check();

      // Enable a plugin checkbox
      // (e.g., Delete revisions to maintain a minimum count).
      $details = $page->find('css', "details#edit-{$entity_type_name}-settings-amount");
      $summary = $details->find('xpath', './summary');
      $summary->click();
      $edit["{$entity_type}[settings][amount][status]"] = TRUE;
      $input = $page->findField("{$entity_type}[settings][amount][settings][amount]");
      // Check for empty value.
      $input->setValue('');
      $this->submitForm($edit, 'edit-submit');
      $page = $this->getSession()->getPage();
      $amount = $page->findField("{$entity_type}[settings][amount][settings][amount]");
      $this->assertTrue($amount->hasAttribute('required'), 'Please fill out this field.');

      // Check for negative value.
      $page = $this->getSession()->getPage();
      $input->setValue('-4');
      $this->submitForm($edit, 'edit-submit');
      $page = $this->getSession()->getPage();
      $this->assertTrue($amount->hasAttribute('required'), 'Value must be greater than or equal to 1.');
    }

    // Check form validations on revision manager
    // Node bundle configuration form.
    $this->drupalGet(self::REVISION_MANAGER_PATH);
    $page = $this->getSession()->getPage();
    $details = $this->getSession()->getPage()->find('css', "form#revision-manager-settings #edit-node");
    $summary = $details->find('xpath', './summary');
    $summary->click();
    $edit = [];
    $edit['node[enable_entity_type]'] = TRUE;
    $this->submitForm($edit, 'edit-submit');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');
    $this->drupalGet('/admin/structure/types/manage/page');
    $page = $this->getSession()->getPage();
    $tab = $page->findLink('Revision Manager');
    $this->assertNotNull($tab, 'Revision Manager tab found.');
    $tab->click();
    $details = $page->find('css', "details#edit-revision-manager-node-amount");
    $summary = $details->find('xpath', './summary');
    $summary->click();
    $edit = ['revision_manager[node][amount][status]' => TRUE];
    $input = $page->findField("revision_manager[node][amount][settings][amount]");
    // Check for empty value.
    $input->setValue('');
    $this->submitForm($edit, 'edit-submit');
    $page = $this->getSession()->getPage();
    $amount = $page->findField("revision_manager[node][amount][settings][amount]");
    $this->assertTrue($amount->hasAttribute('required'), 'Please fill out this field.');

    // Check for negative value.
    $page = $this->getSession()->getPage();
    $input->setValue('-4');
    $this->submitForm($edit, 'edit-submit');
    $page = $this->getSession()->getPage();
    $this->assertTrue($amount->hasAttribute('required'), 'Value must be greater than or equal to 1.');
  }

  /**
   * Test Revision Manager Enqueue now.
   */
  public function testPluginEnabledButQueueingDisabled(): void {

    $queue = \Drupal::service('queue')->get('revision_manager_remove_revisions');
    $queue->deleteQueue();
    Node::create(['type' => 'page', 'title' => "Test Page 1"])->save();
    Media::create(['bundle' => 'test_image', 'name' => 'Test Media 1'])->save();
    Term::create(['vid' => 'test', 'name' => 'Test taxonomy 1'])->save();
    BlockContent::create(['type' => 'basic_block', 'info' => 'Test Block 1'])->save();
    // Go to revision manager config.
    $this->drupalGet(self::REVISION_MANAGER_PATH);

    // Loop over supported entity types.
    $entity_types = [
      'node' => 1,
      'media' => 1,
      'taxonomy_term' => 1,
      'block_content' => 1,
    ];

    foreach ($entity_types as $entity_type => $count) {
      $entity_type_name = str_replace('_', '-', $entity_type);
      $details = $this->getSession()->getPage()->find('css', "form#revision-manager-settings #edit-{$entity_type_name}");
      $summary = $details->find('xpath', './summary');
      $summary->click();
      $edit["{$entity_type}[enable_entity_type]"] = TRUE;
    }
    $edit["disable_automatic_queueing"] = TRUE;
    $this->submitForm($edit, 'edit-submit');
    $this->assertSession()->waitForText('The configuration options have been saved.', 5000);
    // Check that queue is empty.
    $queue = \Drupal::service('queue')->get('revision_manager_remove_revisions');
    $this->assertEquals(0, $queue->numberOfItems(), 'No queue items created when queueing is disabled.');
  }

  /**
   * Test Bundle Enqueue now.
   */
  public function testBundleEnqueue(): void {
    $queue = \Drupal::service('queue')->get('revision_manager_remove_revisions');
    $queue->deleteQueue();

    // Go to revision manager config
    // and enable queueing for node entity type.
    $this->drupalGet(self::REVISION_MANAGER_PATH);
    $page = $this->getSession()->getPage();
    $details = $page->find('css', "form#revision-manager-settings #edit-node");
    $summary = $details->find('xpath', './summary');
    $summary->click();

    $edit = ['node[enable_entity_type]' => TRUE];
    $this->submitForm($edit, 'edit-submit');
    $this->getSession()->wait(1000);
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Create test entities AFTER enabling the config.
    Node::create(['type' => 'page', 'title' => "Test Page 1"])->save();

    // Go to content type configuration screen.
    $this->drupalGet('/admin/structure/types/manage/page');
    $page = $this->getSession()->getPage();
    $tab = $page->findLink('Revision Manager');
    $this->assertNotNull($tab, 'Revision Manager tab found.');
    $tab->click();
    $this->assertSession()->waitForElementVisible('css', '#edit-revision-manager-node-amount-status', 5000);
    $page->find('css', 'details#edit-revision-manager-node-amount summary')->click();
    $this->submitForm([
      'revision_manager[node][amount][status]' => TRUE,
      'enqueue_entities_now' => TRUE,
    ], 'edit-submit');
    $this->getSession()->wait(5000);

    // Reload queue and check item count.
    $queue = \Drupal::service('queue')->get('revision_manager_remove_revisions');
    $actual_count = $queue->numberOfItems();
    $this->assertEquals(1, $actual_count, "Expected 1 queue item, found: $actual_count");
  }

  /**
   * Creates a media type with an image source field.
   *
   * @param string $id
   *   The machine name of the media type.
   * @param string $label
   *   The human-readable name of the media type.
   * @param string $field_name
   *   The image field name (default: field_media_image).
   */
  protected function createMediaTypeWithImageField(string $id = 'test_image', string $label = 'Test Image', string $field_name = 'field_media_image'): void {
    // Create the media type first.
    if (!MediaType::load($id)) {
      MediaType::create([
        'id' => $id,
        'label' => $label,
        'source' => 'image',
        'source_configuration' => [
          'source_field' => $field_name,
        ],
      ])->save();
    }

    // Create field storage if it doesn't exist.
    if (!FieldStorageConfig::loadByName('media', $field_name)) {
      FieldStorageConfig::create([
        'field_name' => $field_name,
        'entity_type' => 'media',
        'type' => 'image',
        'cardinality' => 1,
        'translatable' => TRUE,
      ])->save();
    }

    // Attach field to the media bundle.
    if (!FieldConfig::loadByName('media', $id, $field_name)) {
      FieldConfig::create([
        'field_name' => $field_name,
        'entity_type' => 'media',
        'bundle' => $id,
        'label' => 'Image',
      ])->save();
    }

    // Optional: Set the form display.
    \Drupal::service('entity_display.repository')
      ->getFormDisplay('media', $id, 'default')
      ->setComponent($field_name, [
        'type' => 'image_image',
      ])
      ->save();

    // Optional: Set the view display.
    \Drupal::service('entity_display.repository')
      ->getViewDisplay('media', $id, 'default')
      ->setComponent($field_name, [
        'type' => 'image',
      ])
      ->save();
  }

}
