<?php

namespace Drupal\Tests\revision_manager\Kernel;

use Drupal\Core\Entity\RevisionableInterface;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\Tests\revision_manager\Traits\TestRevisionManagerTrait;

/**
 * Tests Revision Manager plugins on canvas_page entities.
 *
 * @group revision_manager
 * @group canvas
 * @requires module canvas
 * @requires extension pdo_sqlite
 */
class CanvasPageEntityTest extends EntityKernelTestBase {

  use TestRevisionManagerTrait;

  /**
   * Queue used for Revision Manager revision deletion.
   */
  public const QUEUE_ID = 'revision_manager_remove_revisions';

  /**
   * Time constants.
   */
  protected const ONE_DAY = 86_400;
  protected const ONE_WEEK = 604_800;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'ckeditor5',
    'datetime',
    'editor',
    'field',
    'file',
    'filter',
    'image',
    'link',
    'media',
    'media_library',
    'options',
    'path',
    'path_alias',
    'revision_manager',
    'system',
    'text',
    'user',
    'views',
    'canvas',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('canvas_page');
    $this->installEntitySchema('path_alias');
    $this->installConfig(['canvas']);
    $this->container->get('router.builder')->rebuild();
    $this->container->get('plugin.manager.revision_manager')->clearCachedDefinitions();
  }

  /**
   * Test amount plugin on canvas page entities.
   */
  public function testAmountPluginCanvasPage(): void {
    $this->configureRevisionManager('canvas_page', 'amount', TRUE, ['amount' => 4]);
    $entity = $this->createEntity('canvas_page', [
      'title' => 'Canvas Page Amount Test',
      'status' => 1,
    ]);
    $this->assertInstanceOf(RevisionableInterface::class, $entity);

    $this->createRevisions($entity, 8);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify exactly 4 revisions remain.
    $this->assertRevisionCount($entity, 4);
  }

  /**
   * Test age plugin on canvas page entities.
   */
  public function testAgePluginCanvasPage(): void {
    $this->configureRevisionManager('canvas_page', 'age', TRUE, ['age' => 2]);
    $entity = $this->createEntity('canvas_page', [
      'title' => 'Canvas Page Age Test',
      'status' => 1,
    ]);
    $this->assertInstanceOf(RevisionableInterface::class, $entity);

    $cutoff = (new \DateTimeImmutable())->modify('-2 months')->getTimestamp();

    // Create revisions before and after the cutoff date.
    $this->createRevisionsAroundCutoff(
      $entity, 5, 2, $cutoff,
      self::ONE_WEEK * 2,
      self::ONE_DAY * 3,
    );

    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify only the original + 2 newer revisions remain.
    $this->assertRevisionCount($entity, 3);
  }

}
