<?php

namespace Drupal\Tests\revision_manager\Kernel;

use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Entity\TranslatableInterface;

/**
 * Tests multilingual revision preservation.
 *
 * @group revision_manager
 */
class MultilingualPluginTest extends RevisionManagerTestBase {

  /**
   * Test single translation keeps configured amount.
   */
  public function testSingleTranslation(): void {
    $this->configureRevisionManager('node', 'amount', TRUE, ['amount' => 3]);
    $node = $this->createEntity('node', ['type' => 'page', 'title' => 'EN Original', 'status' => 1]);

    $this->assertInstanceOf(TranslatableInterface::class, $node);
    $this->assertInstanceOf(RevisionableInterface::class, $node);

    $translation = $node->addTranslation('es', ['title' => 'ES']);
    $translation->setNewRevision(TRUE);
    $translation->save();
    $node = $this->reloadEntity($node);

    $this->createTranslationRevisions($node, 'es', 4);
    $this->assertRevisionCount($node, 6);

    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify 3 ES + 1 EN = 4 revisions remaining.
    $this->assertRevisionCount($node, 4);
  }

  /**
   * Test revisions are kept per translation, not per entity.
   */
  public function testMultipleTranslationsKeepRevisionsPerLanguage(): void {
    $this->configureRevisionManager('node', 'amount', TRUE, ['amount' => 2]);
    $node = $this->createEntity('node', ['type' => 'page', 'title' => 'EN Original', 'status' => 1]);

    $translation = $node->addTranslation('de', ['title' => 'DE Translation']);
    $translation->setNewRevision(TRUE);
    $translation->save();
    $node = $this->reloadEntity($node);

    $this->createTranslationRevisions($node, 'en', 3);
    $node = $this->reloadEntity($node);

    $this->createTranslationRevisions($node, 'de', 3);
    $node = $this->reloadEntity($node);

    $this->assertRevisionCount($node, 8);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify 2 EN + 2 DE = 4 revisions remaining.
    $this->assertRevisionCount($node, 4);
  }

  /**
   * Creates revisions for a specific translation.
   */
  protected function createTranslationRevisions(TranslatableInterface $entity, string $langcode, int $count): void {
    $storage = $this->container->get('entity_type.manager')->getStorage($entity->getEntityTypeId());

    for ($i = 0; $i < $count; $i++) {
      $entity = $storage->load($entity->id());
      $translation = $entity->getTranslation($langcode);
      $translation->setNewRevision(TRUE);
      $translation->isDefaultRevision(TRUE);
      $translation->set('title', $translation->label() . ' rev' . ($i + 1));
      $translation->save();
    }
  }

}
