<?php

namespace Drupal\Tests\revision_manager\Kernel;

/**
 * Tests queue generation logic for Revision Manager.
 *
 * @group revision_manager
 */
class QueueGeneratorTest extends RevisionManagerTestBase {

  /**
   * The queue generator service.
   *
   * @var \Drupal\revision_manager\Manager\QueueGenerator
   */
  protected $generator;

  /**
   * The queue for remove revisions.
   *
   * @var \Drupal\Core\Queue\QueueInterface
   */
  protected $queue;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Enable the node entity type for queue tests.
    $this->configureRevisionManager('node', 'amount', TRUE);

    $this->generator = $this->container->get('revision_manager.queue_generator');
    $this->queue = $this->container->get('queue')->get(self::QUEUE_ID);

    // Clear any existing queue items and state.
    $this->generator->clearQueue();
  }

  /**
   * Test enqueueEntity adds an item only once per entity.
   */
  public function testEnqueueEntityIsUnique(): void {
    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Unique', 'status' => 1]);
    $id = (string) $entity->id();

    // First enqueue should succeed.
    $this->assertTrue($this->generator->enqueueEntity('node', $id));
    $this->assertEquals(1, $this->queue->numberOfItems());

    // Second enqueue should be ignored.
    $this->assertFalse($this->generator->enqueueEntity('node', $id));
    $this->assertEquals(1, $this->queue->numberOfItems());
  }

  /**
   * Test clearQueue empties the queue and resets the state.
   */
  public function testClearQueue(): void {
    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Clear', 'status' => 1]);
    $id = (string) $entity->id();
    $this->generator->enqueueEntity('node', $id);

    $this->generator->clearQueue();
    $this->assertEquals(0, $this->queue->numberOfItems());

    // After clearing, enqueue should succeed again.
    $this->assertTrue($this->generator->enqueueEntity('node', $id));
  }

  /**
   * Test enqueueRevisionsByBundle enqueues all entities in a bundle.
   */
  public function testEnqueueByBundle(): void {
    // Create multiple nodes in the 'page' bundle.
    $this->createEntity('node', ['type' => 'page', 'title' => 'One', 'status' => 1]);
    $this->createEntity('node', ['type' => 'page', 'title' => 'Two', 'status' => 1]);

    $this->generator->enqueueRevisionsByBundle('node', 'page');
    $this->assertEquals(2, $this->queue->numberOfItems());
  }

  /**
   * Test enqueueRevisionsByType enqueues all entities across bundles.
   */
  public function testEnqueueByType(): void {
    // Create nodes in the 'page' and 'article' bundle.
    $this->createEntity('node', ['type' => 'page', 'title' => 'Page Bundle', 'status' => 1]);
    $this->createEntity('node', ['type' => 'article', 'title' => 'Article', 'status' => 1]);

    $this->generator->enqueueRevisionsByType('node');
    $this->assertEquals(2, $this->queue->numberOfItems());
  }

  /**
   * Test disabled entity types are not enqueued.
   */
  public function testDisabledEntityTypeDoesNotEnqueue(): void {
    // Disable the node entity type in config.
    $this->configureRevisionManager('node', 'amount', FALSE);

    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Not Queued', 'status' => 1]);
    $this->assertFalse($this->generator->enqueueEntity('node', (string) $entity->id()));
    $this->assertEquals(0, $this->queue->numberOfItems());
  }

  /**
   * Test Group entities can be enumerated and enqueued.
   *
   * This specifically tests that EntityHelper::getEntityIds() works with
   * Group entities. Group module has complex access control that previously
   * caused issues when accessCheck(TRUE) was used.
   *
   * @see https://www.drupal.org/project/revision_manager/issues/XXXXXXX
   */
  public function testEnqueueGroupEntities(): void {
    $this->configureRevisionManager('group', 'amount', TRUE, ['amount' => 5]);

    // Create multiple Group entities.
    $this->createEntity('group', ['type' => 'test_group_type', 'label' => 'Group 1', 'status' => 1]);
    $this->createEntity('group', ['type' => 'test_group_type', 'label' => 'Group 2', 'status' => 1]);
    $this->createEntity('group', ['type' => 'test_group_type', 'label' => 'Group 3', 'status' => 1]);

    // Test EntityHelper can find all Group entities.
    $entity_helper = $this->container->get('revision_manager.entity_helper');
    $ids = $entity_helper->getEntityIds('group', 'test_group_type');
    $this->assertCount(3, $ids, 'EntityHelper should find all 3 Group entities');

    // Test the batch enumeration flow works.
    $this->generator->enqueueRevisionsByBundle('group', 'test_group_type');
    $this->assertEquals(3, $this->queue->numberOfItems(), 'All 3 Group entities should be enqueued');
  }

  /**
   * Test enqueueRevisionsByType works for Group entities.
   */
  public function testEnqueueGroupsByType(): void {
    $this->configureRevisionManager('group', 'amount', TRUE, ['amount' => 5]);

    $this->createEntity('group', ['type' => 'test_group_type', 'label' => 'Group A', 'status' => 1]);
    $this->createEntity('group', ['type' => 'test_group_type', 'label' => 'Group B', 'status' => 1]);

    $this->generator->enqueueRevisionsByType('group');
    $this->assertEquals(2, $this->queue->numberOfItems());
  }

}
