<?php

declare(strict_types=1);

namespace Drupal\revision_purgatory\Commands;

use Drupal\revision_purgatory\Services\RevisionPurgatoryRandomRevisionGenerator;
use Drush\Commands\DrushCommands;

/**
 * Drush commands for Revision Purgatory helpers.
 */
class RevisionPurgatoryCommands extends DrushCommands {

  /**
   * Random revision generator service.
   */
  protected RevisionPurgatoryRandomRevisionGenerator $generator;

  /**
   * Constructs the command class.
   */
  public function __construct(RevisionPurgatoryRandomRevisionGenerator $generator) {
    parent::__construct();
    $this->generator = $generator;
  }

  /**
   * Generates random node revisions for testing.
   *
   * @param int $nid
   *   Node ID to mutate.
   * @param int $count
   *   Number of revisions to create.
   * @param array $options
   *   Command options.
   *
   * @option start
   *   Lower timestamp bound (Unix).
   * @option end
   *   Upper timestamp bound (Unix).
   * @option languages
   *   Comma separated language codes to target. Defaults to the node language.
   *
   * @command revision-purgatory:generate-revisions
   * @aliases rp-gr,revision-purgatory:rand-revs
   *
   * @usage drush revision-purgatory:generate-revisions 10 5
   *   Create five revisions between now and 90 days ago for node 10.
   * @usage drush revision-purgatory:generate-revisions 10 5 --start=1700000000 --end=1700500000
   *   Create revisions for node 10 within a specific timestamp window.
   * @usage drush revision-purgatory:generate-revisions 10 3 --languages=en,fr
   *   Create three revisions per language for the EN and FR translations.
   */
  public function generateRandomRevisions(int $nid, int $count = 1, array $options = ['start' => null, 'end' => null, 'languages' => null]): void {
    $start = isset($options['start']) ? (int) $options['start'] : null;
    $end = isset($options['end']) ? (int) $options['end'] : null;
    $langcodes = [];
    if (!empty($options['languages'])) {
      $langcodes = array_values(array_filter(array_map('trim', explode(',', (string) $options['languages']))));
    }

    try {
      $revision_ids = $this->generator->generate($nid, $count, $start, $end, $langcodes);
    }
    catch (\Throwable $exception) {
      throw new \RuntimeException($exception->getMessage(), $exception->getCode(), $exception);
    }

    $this->logger()->success(dt('Generated @count revisions. Latest revision ID: @vid', [
      '@count' => count($revision_ids),
      '@vid' => end($revision_ids) ?: 'n/a',
    ]));
  }

}
