<?php

declare(strict_types=1);

namespace Drupal\revision_purgatory\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\revision_purgatory\Services\RevisionPurgatoryRevisionService;
use Drupal\revision_purgatory\Services\UrlNormalizerService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Displays and processes node version removal form.
 */
class RevisionPurgatoryVersionListForm extends FormBase {

  private const DEFAULT_BATCH_CHUNK_SIZE = 5;

  /**
   * Revision retrieval service.
   *
   * @var \Drupal\revision_purgatory\Services\RevisionPurgatoryRevisionService
   */
  protected RevisionPurgatoryRevisionService $revisionService;

  /**
   * URL normalization service.
   *
   * @var \Drupal\revision_purgatory\Services\UrlNormalizerService
   */
  protected UrlNormalizerService $urlNormalizer;

  /**
   * Config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactoryService;

  /**
   * Date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * Constructs the version list form.
   *
   * @param \Drupal\revision_purgatory\Services\RevisionPurgatoryRevisionService $revision_service
   *   The revision helper service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\revision_purgatory\Services\UrlNormalizerService $url_normalizer
   *   The URL normalization helper.
   */
  public function __construct(
    RevisionPurgatoryRevisionService $revision_service,
    ConfigFactoryInterface $config_factory,
    DateFormatterInterface $date_formatter,
    UrlNormalizerService $url_normalizer
  ) {
    $this->revisionService = $revision_service;
    $this->configFactoryService = $config_factory;
    $this->dateFormatter = $date_formatter;
    $this->urlNormalizer = $url_normalizer;
  }

  /**
   * Creates the form instance from the container.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   *
   * @return static
   *   The form instance.
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('revision_purgatory.revision_service'),
      $container->get('config.factory'),
      $container->get('date.formatter'),
      $container->get('revision_purgatory.url_normalizer')
    );
  }

  /**
   * Returns the form ID.
   *
   * @return string
   *   The unique form identifier.
   */
  public function getFormId(): string {
    return 'revision_purgatory_version_list_form';
  }

  /**
   * Builds a consistently formatted cell value for the versions table.
   *
   * @param string $langcode
   *   The language code used to determine grouped rows.
   * @param string|null $cell
   *   The raw cell value retrieved from the revision query.
   * @param string|null $type
   *   Optional formatter type (e.g. 'date' or 'status').
   *
   * @return string
   *   The formatted string ready for display.
   */
  public function formattedCell(string $langcode, ?string $cell, string $type = null): string {
    $formatted_cell = $cell;

    switch ($type) {
      case 'date':
        $formatted_cell = $this->dateFormatter->format((int) $cell, 'custom', 'Y-m-d H:i');
        break;

      case 'status':
        $formatted_cell = (int) $cell === 1 ? $this->t('Published') : $this->t('Non Published State');
        break;
    }

    return (string) $formatted_cell;
  }

  /**
   * Builds the version listing form.
   *
   * @param array $form
   *   The base form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   * @param int|null $nid
   *   (optional) The node ID being managed.
   * @param string|null $langcode
   *   (optional) The language code filter.
   *
   * @return array
   *   The complete form definition.
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?int $nid = null, ?string $langcode = null): array {
    $revisions = $this->revisionService->getVersionsBy($nid, (string) $langcode);
    $query_params = $this->urlNormalizer->currentQuery();
    $back_link = Link::fromTextAndUrl(
      $this->t('← Back to Node List'),
      Url::fromRoute('revision_purgatory.nodes', [], [
          'query' => $this->urlNormalizer->forBackLink($query_params)
      ])
    )->toString();

    $rows = [];
    foreach ($revisions as $revision) {
      $rows[] = [
        'id' => $revision->vid,
        'langcode' => $this->formattedCell($langcode, $revision->langcode ?? ''),
        'title' => $this->formattedCell($langcode, $revision->title ?? ''),
        'status' => $this->formattedCell($langcode, $revision->status ?? '', 'status'),
        'created' => $this->formattedCell($langcode, $revision->created ?? '', 'date'),
        'changed' => $this->formattedCell($langcode, $revision->changed ?? '', 'date'),
      ];
    }

    $header = [
      [
        'data' => [
          '#type' => 'html_tag',
          '#tag' => 'input',
          '#attributes' => [
            'type' => 'checkbox',
            'id' => 'edit-table-select-all',
            'title' => $this->t('Select all'),
          ],
        ],
        'class' => ['select-all-checkbox'],
        'style' => 'text-align: center;',
      ],
      'Version ID' => $this->t('Version ID'),
      'Langcode' => $this->t('Langcode'),
      'Title' => $this->t('Title'),
      'Status' => $this->t('Status'),
      'Created' => $this->t('Created'),
      'Changed' => $this->t('Changed'),
    ];

    $form['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#empty' => $this->t('No entries found.'),
      '#prefix' => $back_link,
      '#attributes' => [
        'id' => 'revision-purgatory-version-list-table',
      ],
    ];

    foreach ($rows as $key => $row) {
      $form['table'][$key]['checkbox'] = [
        '#type' => 'checkbox',
        '#return_value' => $row['id'],
        '#default_value' => 0,
        '#attributes' => [
          'class' => ['revision-purgatory-row-checkbox'],
        ],
      ];
      $form['table'][$key]['id'] = [
        [
          '#type' => 'link',
          '#title' => $row['id'],
          '#url' => Url::fromRoute(
            'revision_purgatory.node_revisions',
            [
              'nid' => $nid,
              'vid' => $row['id'],
              'langcode' => $langcode,
            ],
            ['query' => $this->urlNormalizer->forDetailLink($query_params)]
          ),
        ],
      ];
      $form['table'][$key]['langcode'] = [
        '#plain_text' => $row['langcode'],
      ];
      $form['table'][$key]['title'] = [
        '#plain_text' => $row['title'],
      ];
      $form['table'][$key]['status'] = [
        '#plain_text' => $row['status'],
      ];
      $form['table'][$key]['created'] = [
        '#plain_text' => $row['created'],
      ];
      $form['table'][$key]['changed'] = [
        '#plain_text' => $row['changed'],
      ];
    }

    $form['pager'] = [
      '#type' => 'pager',
    ];

    $form['#attached']['library'][] = 'revision_purgatory/revision_purgatory_select_all';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Remove selected'),
    ];

    return $form;
  }

  /**
   * Processes the submitted form.
   *
   * @param array $form
   *   The submitted form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The submitted form state.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Second argument in url is langcode.
    $build_info = $form_state->getBuildInfo();
    $arguments = $build_info['args'] ?? [];
    $langcode = (string) ($arguments[1] ?? '');

    $values = $form_state->getValue('table');
    $selected = [];

    foreach ($values as $row_id => $row) {
      if (!empty($row['checkbox'])) {
        $selected[] = $row['checkbox'];
      }
    }

    $config = $this->configFactoryService->get('revision_purgatory.settings');
    $chunk_size = (int) $config->get('batch_chunk_size') ?: self::DEFAULT_BATCH_CHUNK_SIZE;
    $chunks = array_chunk($selected, $chunk_size);

    $operations = [];
    foreach ($chunks as $chunk) {
      $operations[] = [
        [get_class($this), 'batchProcess'],
        [$chunk, $langcode],
      ];
    }

    $batch = [
      'title' => $this->t('Removing versions...'),
      'operations' => $operations,
      'finished' => [get_class($this), 'batchFinished'],
    ];

    batch_set($batch);
  }

  /**
   * Batch callback for deleting versions.
   *
   * @param array $ids
   *   The revision IDs to delete.
   * @param string|null $langcode
   *   The language code applied to the deletions.
   * @param array $context
   *   Batch context array.
   */
  public static function batchProcess(array $ids, ?string $langcode, array &$context): void {
    foreach ($ids as $id) {
      $rev_service = \Drupal::service('revision_purgatory.revision_service');
      $rev_service->deleteVersion((int) $id, $langcode);
    }

    $context['message'] = t('Removed IDs: @ids', ['@ids' => implode(', ', $ids)]);
  }

  /**
   * Finishes the batch process.
   *
   * @param bool $success
   *   True if the batch completed successfully.
   * @param array $results
   *   The results array.
   * @param array $operations
   *   The remaining operations array.
   */
  public static function batchFinished(bool $success, array $results, array $operations): void {
    if ($success) {
      \Drupal::messenger()->addMessage(t('Removal processing complete. Check message log in case there are any issues.'));
    }
    else {
      \Drupal::messenger()->addError(t('An error occurred during batch processing.'));
    }
  }

}
