<?php

declare(strict_types=1);

namespace Drupal\revision_purgatory\Plugin\QueueWorker;

use Drupal\Core\Annotation\Translation;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\revision_purgatory\Services\RevisionPurgatoryRevisionService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes items for the Revision Purgatory queue.
 *
 * @QueueWorker(
 *   id = "revision_purgatory_queue",
 *   title = @Translation("Revision Purgatory Queue Worker"),
 *   cron = {"time" = 90}
 * )
 */
class RevisionPurgatoryQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Revision service used to delete versions.
   *
   * @var \Drupal\revision_purgatory\Services\RevisionPurgatoryRevisionService
   */
  protected RevisionPurgatoryRevisionService $revisionService;

  /**
   * Logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * Recently processed revisions.
   *
   * @var string[]
   */
  protected array $processedItems = [];

  /**
   * Number of processed items to log at once.
   *
   * @var int
   */
  protected int $batchLogSize = 1;

  /**
   * Queue factory for queue statistics.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected QueueFactory $queueFactory;

  /**
   * Constructs the queue worker.
   *
   * @param array $configuration
   *   Configuration array for the plugin instance.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\revision_purgatory\Services\RevisionPurgatoryRevisionService $revision_service
   *   The revision purge helper service.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   Logger channel for audit output.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory for logging thresholds.
   * @param \Drupal\Core\Queue\QueueFactory $queue_factory
   *   Queue factory for retrieving queue stats.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    RevisionPurgatoryRevisionService $revision_service,
    LoggerChannelInterface $logger,
    ConfigFactoryInterface $config_factory,
    QueueFactory $queue_factory
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->revisionService = $revision_service;
    $this->logger = $logger;
    $this->queueFactory = $queue_factory;
    $config = $config_factory->get('revision_purgatory.settings');
    $this->batchLogSize = (int) $config->get('batch_chunk_size');
  }

  /**
   * Creates the queue worker via the service container.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   * @param array $configuration
   *   Plugin configuration array.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   *
   * @return static
   *   The instantiated queue worker.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition
  ) {

    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('revision_purgatory.revision_service'),
      $container->get('logger.factory')->get('revision_purgatory'),
      $container->get('config.factory'),
      $container->get('queue')
    );
  }

  /**
   * Processes a single queue item payload.
   *
   * @param array $data
   *   The queue item data array.
   *
   * @return void
   */
  public function processItem($data): void {
    $this->revisionService->deleteVersion((int) $data['vid']);
    $this->processedItems[] = sprintf('%d:%d', (int) $data['nid'], (int) $data['vid']);
    $remaining_items = $this->queueFactory->get('revision_purgatory_queue')->numberOfItems();
    // We count until 1, because numberOfItems is not processed yet.
    if (count($this->processedItems) >= $this->batchLogSize || $remaining_items <= 1) {
      $this->flushProcessedLog();
    }
  }

  /**
   * Writes the audit log and resets internal buffers.
   *
   * @return void
   */
  protected function flushProcessedLog(): void {
    if (!$this->processedItems) {
      return;
    }

    $this->logger->info('Removed revisions: @items', [
      '@items' => implode(', ', $this->processedItems),
    ]);
    $this->processedItems = [];
  }
}
