<?php

declare(strict_types=1);

namespace Drupal\revision_purgatory\Services;

use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Normalizes pager query context for list/detail routes.
 */
class UrlNormalizerService {

  /**
   * Constructs the URL normalizer.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack used to read current query data.
   */
  public function __construct(
    protected RequestStack $requestStack
  ) {
  }

  /**
   * Base query from the current request.
   *
   * @return array
   *   The copied query parameters.
   */
  public function currentQuery(): array {
    return $this->requestStack->getCurrentRequest()?->query->all() ?? [];
  }

  /**
   * Query for entering a detail route (strip primary pager).
   *
   * @param array $query
   *   Existing query parameters.
   *
   * @return array
   *   Modified query parameters with primary pager removed.
   */
  public function forDetailLink(array $query): array {
    if (isset($query['page'])) {
      if (!isset($query['_parent_page'])) {
        $query['_parent_page'] = $query['page'];
      }
      unset($query['page']);
    }

    return $query;
  }

  /**
   * Query for returning to list routes (restore parent pager).
   *
   * @param array $query
   *   Existing query parameters.
   *
   * @return array
   *   Modified query parameters with the parent pager restored.
   */
  public function forBackLink(array $query): array {
    if (isset($query['_parent_page'])) {
      $query['page'] = $query['_parent_page'];
      unset($query['_parent_page']);
    }

    return $query;
  }

}
