<?php

declare(strict_types=1);

namespace Drupal\revision_purgatory\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Builds filter form for node revision lists.
 */
class NodeRevisionContentTypeFilterForm extends FormBase {

  /**
   * Entity type manager for loading content types.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs the filter form.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Creates the filter form instance.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   *
   * @return static
   *   The form instance.
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * Returns the unique form ID.
   *
   * @return string
   *   The form identifier.
   */
  public function getFormId(): string {
    return 'node_revision_content_type_filter_form';
  }

  /**
   * Builds the filter form structure.
   *
   * @param array $form
   *   The existing form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   *
   * @return array
   *   The completed form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $content_types = $this->entityTypeManager->getStorage('node_type')->loadMultiple();
    $options = ['' => $this->t('- All -')];
    foreach ($content_types as $type) {
      $options[$type->id()] = $type->label();
    }

    $request = $this->getRequest();
    $content_type_default = $request->query->get('content_type', '');
    $title_default = $request->query->get('title', '');

    $form['content_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Content type'),
      '#options' => $options,
      '#default_value' => $content_type_default,
    ];
    $form['title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Title contains'),
      '#default_value' => $title_default,
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Filter'),
    ];

    return $form;
  }

  /**
   * Handles submission of the filter form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $content_type = $form_state->getValue('content_type');
    $title = $form_state->getValue('title');
    $form_state->setRedirect('revision_purgatory.nodes', [], [
      'query' => [
        'content_type' => $content_type,
        'title' => $title,
      ],
    ]);
  }
}
