<?php

declare(strict_types=1);

namespace Drupal\revision_purgatory\Services;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Entity\ContentEntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\node\NodeInterface;

/**
 * Generates random node revisions for testing purge scenarios.
 */
class RevisionPurgatoryRandomRevisionGenerator {

  /**
   * Node storage.
   *
   * @var \Drupal\Core\Entity\ContentEntityStorageInterface
   */
  protected ContentEntityStorageInterface $nodeStorage;

  /**
   * Time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected TimeInterface $time;

  /**
   * Creates the generator.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager, TimeInterface $time) {
    $this->nodeStorage = $entityTypeManager->getStorage('node');
    $this->time = $time;
  }

  /**
   * Creates random revisions for a node within a timestamp window.
   *
   * @param int $nid
   *   Node ID to mutate.
   * @param int $count
   *   Number of revisions to create.
   * @param int|null $startTimestamp
   *   Lower timestamp bound (defaults to 90 days ago).
   * @param int|null $endTimestamp
   *   Upper timestamp bound (defaults to now).
   * @param array $langcodes
   *   Optional list of language codes to generate revisions for. Defaults to the
   *   node's current language.
   *
   * @return array
   *   Generated revision IDs.
   */
  public function generate(int $nid, int $count, ?int $startTimestamp = null, ?int $endTimestamp = null, array $langcodes = []): array {
    if ($count <= 0) {
      throw new \InvalidArgumentException('The revision count must be greater than zero.');
    }

    /** @var \Drupal\node\NodeInterface|null $node */
    $node = $this->nodeStorage->load($nid);
    if (!$node instanceof NodeInterface) {
      throw new \InvalidArgumentException(sprintf('Node %d could not be loaded.', $nid));
    }

    $node_created = (int) $node->getCreatedTime();
    $end = $endTimestamp ?? $this->time->getRequestTime();
    $end = max($end, $node_created);
    $start = $startTimestamp ?? ($end - (60 * 60 * 24 * 90));

    if ($start > $end) {
      [$start, $end] = [$end, $start];
    }
    $start = max($start, $node_created);

    $langcodes = array_values(array_filter(array_unique($langcodes)));
    if (empty($langcodes)) {
      $langcodes = [$node->language()->getId()];
    }

    foreach ($langcodes as $langcode) {
      if (!$node->hasTranslation($langcode)) {
        throw new \InvalidArgumentException(sprintf('Node %d does not have the %s translation.', $nid, $langcode));
      }
    }

    $revision_ids = [];
    foreach ($langcodes as $langcode) {
      for ($i = 0; $i < $count; $i++) {
        $timestamp = random_int($start, $end);
        $revision_ids[] = $this->createRevision($node, $timestamp, $i + 1, $langcode);
        // Reload to ensure the next revision starts from the latest base entity.
        $node = $this->nodeStorage->load($nid);
      }
    }

    return $revision_ids;
  }

  /**
   * Creates a single revision with a specific timestamp.
   *
   * @param \Drupal\node\NodeInterface $node
   *   Base node entity.
   * @param int $timestamp
   *   Revision creation timestamp.
   * @param int $sequence
   *   Sequential counter for log messages.
   * @param string $langcode
   *   Language code to mutate.
   *
   * @return int
   *   The new revision ID.
   */
  protected function createRevision(NodeInterface $node, int $timestamp, int $sequence, string $langcode): int {
    if (!$node->hasTranslation($langcode)) {
      throw new \InvalidArgumentException(sprintf('Node %d no longer has the %s translation.', $node->id(), $langcode));
    }

    /** @var \Drupal\node\NodeInterface $translation */
    $translation = $node->getTranslation($langcode);
    $translation->setNewRevision(true);
    $translation->isDefaultRevision(true);
    $translation->setRevisionCreationTime($timestamp);
    $translation->setChangedTime($timestamp);
    $translation->setRevisionTranslationAffected(true);
    $translation->setRevisionLogMessage(sprintf('Random %s revision %d generated at %d.', $langcode, $sequence, $timestamp));
    $translation->save();

    return (int) $translation->getRevisionId();
  }

}
