<?php

declare(strict_types=1);

namespace Drupal\revision_purgatory\Services;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\StatementInterface;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\ContentEntityStorageInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Queue\QueueFactory;
use IteratorAggregate;

/**
 * Handles revision queries and purge operations for nodes.
 */
class RevisionPurgatoryRevisionService {
  /**
   * Default number of revision rows displayed per paged batch.
   */
  private const REVISION_PAGER_LIMIT = 20;

  /**
   * Entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Entity field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * Database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $connection;

  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Queue factory.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected QueueFactory $queueFactory;

  /**
   * Language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * Node storage accessor.
   *
   * @var \Drupal\Core\Entity\ContentEntityStorageInterface
   */
  protected ContentEntityStorageInterface $storage;

  /**
   * Logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs the service.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager service.
   * @param \Drupal\Core\Queue\QueueFactory $queue_factory
   *   The queue factory.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    Connection $connection,
    ConfigFactoryInterface $config_factory,
    EntityFieldManagerInterface $entity_field_manager,
    QueueFactory $queue_factory,
    LanguageManagerInterface $language_manager,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->connection = $connection;
    $this->configFactory = $config_factory;
    $this->entityFieldManager = $entity_field_manager;
    $this->queueFactory = $queue_factory;
    $this->languageManager = $language_manager;
    $this->logger = $logger_factory->get('revision_purgatory');
  }

  /**
   * Loads revision records for a single revision ID.
   *
   * @param int $vid
   *   The revision ID.
   *
   * @return array
   *   A list of revision row objects.
   */
  public function getRevisionsByVersion(int $vid): array {
    $query = $this->connection->select('node_field_revision', 'nfr')
      ->fields('nfr')
      ->condition('vid', $vid)
      ->orderBy('changed', 'DESC');

    return $query->execute()->fetchAll();
  }

  /**
   * Gets versions (revisions) array.
   *
   * @param int $nid
   *   The node ID.
   *
   * @return array
   *   The revision IDs keyed by revision ID.
   */
  public function getRevisionsBy(int $nid): array {
    $this->storage = $this->entityTypeManager->getStorage('node');
    $node = $this->storage->load($nid);

    $revisions = $this->storage->getQuery()
      ->condition('nid', $node->id())
      ->allRevisions()
      ->accessCheck(false)
      ->execute();

    return $revisions;
  }

  /**
   * Retrieves paged revision entries filtered by language.
   *
   * @param int $nid
   *   The node ID to filter by.
   * @param string $langcode
   *   The language code to filter by, or 'und' for unassigned.
   *
   * @return \Drupal\Core\Database\StatementInterface|\IteratorAggregate
   *   The database statement with revision rows.
   */
  public function getVersionsBy(int $nid, string $langcode): StatementInterface|IteratorAggregate {
    $query = $this->connection->select('node_field_revision', 'nfr')
      ->fields('nfr')
      ->condition('nid', $nid);

    if ($langcode === 'en') {
      $group = $query->orConditionGroup();

      // Case 1: EN translation explicitly affected.
      $group->condition(
        $query->andConditionGroup()
          ->condition('langcode', $langcode)
          ->condition('revision_translation_affected', 1)
      );

      // Case 2: EN translation unaffected, and no other translation is affected.
      $unaffected = $query->andConditionGroup()
        ->condition('langcode', $langcode)
        ->isNull('revision_translation_affected');

      $exists = $this->connection->select('node_field_revision', 'nfr_other')
        ->fields('nfr_other', ['vid'])
        ->where('nfr_other.vid = nfr.vid')
        ->condition('revision_translation_affected', 1);

      $unaffected->notExists($exists);
      $group->condition($unaffected);
      $query->condition($group);
    }
    else {
      $query->condition('langcode', $langcode);
      $query->condition('revision_translation_affected', 1);
    }

    $query->orderBy('changed', 'DESC');
    $pager = $query->extend('Drupal\Core\Database\Query\PagerSelectExtender')
      ->limit(self::REVISION_PAGER_LIMIT);

    return $pager->execute();
  }

  /**
   * Counts revisions per language for a node.
   *
   * NOTE: revision_translation_affected flag can be
   * set for multiple records of the same vid in the database.
   * Example: programmatically created translations during some import.
   * revision_translation_affected flag cannot be set at all for any of the
   * records in the database for the same vid.
   * Example: Translation was removed from the node.
   *
   * @param int $nid
   *   The node ID.
   *
   * @return array
   *   An array of language counts keyed by language code.
   */
  public function getRevisionsPerLanguagesBy(int $nid): array {
    $languages = $this->languageManager->getLanguages();

    $query = $this->connection->select('node_field_revision', 'nfr')
      ->fields('nfr', ['vid', 'langcode', 'revision_translation_affected'])
      ->condition('nid', $nid);
    $results = $query->execute()->fetchAll();

    $lang_revisions = [];
    foreach ($languages as $code => $language) {
      $lang_revisions[$code] = 0;
    }

    $vid_language_affected = [];
    foreach ($results as $row) {
      if (!isset($vid_language_affected[$row->vid])) {
        $vid_language_affected[$row->vid] = 0;
      }

      if ((int) $row->revision_translation_affected === 1) {
        $vid_language_affected[$row->vid] = 1;
        $lang_revisions[$row->langcode]++;
      }
    }

    // Some revisions leave revision_translation_affected flag unset for all rows,
    // so treat them as EN revisions.
    foreach ($vid_language_affected as $row) {
      if (empty($row)) {
        $lang_revisions['en']++;
      }
    }

    return $lang_revisions;
  }

  /**
   * Purges node revisions based on configuration settings.
   *
   * @return int
   *   Number of queue items created.
   */
  public function run(): int {
    $this->storage = $this->entityTypeManager->getStorage('node');
    $config = $this->configFactory->get('revision_purgatory.settings');
    $enable_auto_purge = $config->get('enable_auto_purge');
    $purge_content_types = $config->get('purge_content_types') ?: [];
    $older_than_date = $config->get('purge_older_than');
    $batch_limit = (int) $config->get('batch_chunk_size') ?: 1;

    if (!$enable_auto_purge) {
      return 0;
    }

    $selected_types = array_filter((array) $purge_content_types);
    $older_than = new DrupalDateTime('@' . $older_than_date);

    $query = $this->connection->select('node_revision', 'nr');
    $query->fields('nr', ['nid', 'vid', 'revision_timestamp']);
    $query->join('node', 'n', 'n.nid = nr.nid');
    $query->where('nr.vid <> n.vid');
    $query->condition('nr.revision_timestamp', $older_than->getTimestamp(), '<');
    $query->range(0, $batch_limit);

    if (!empty($selected_types)) {
      $query->condition('n.type', $selected_types, 'IN');
    }

    $revisions = $query->execute()->fetchAllAssoc('vid');

    $queue = $this->queueFactory->get('revision_purgatory_queue');

    foreach ($revisions as $revision) {
      $data = [
        'nid' => (int) $revision->nid,
        'vid' => (int) $revision->vid,
        'revision_timestamp' => (int) $revision->revision_timestamp,
      ];

      $queue->createItem($data);
    }

    return 1;
  }

  /**
   * Deletes a revision by ID.
   *
   * @param int $vid
   *   The revision ID to delete.
   * @param string|null $langcode
   *   (optional) Language context, currently unused.
   */
  public function deleteVersion(int $vid, ?string $langcode = null): void {
    $this->storage = $this->entityTypeManager->getStorage('node');

    try {
      $this->storage->deleteRevision($vid);
    }
    catch (\Throwable $exception) {
      $this->logger->error(
        'Failed to delete revision @vid: @message',
        [
          '@vid' => $vid,
          '@message' => $exception->getMessage(),
          '%exception' => $exception,
        ]
      );
    }
  }
}
