<?php

declare(strict_types=1);

namespace Drupal\Tests\revision_purgatory\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Functional test covering cron-triggered revision purging.
 *
 * @covers \Drupal\revision_purgatory\Services\RevisionPurgatoryRevisionService::run
 * @group revision_purgatory
 */
class RevisionPurgatoryCronTest extends BrowserTestBase {
  private const PUBLISHED_STATUS = 1;

  private const PAST_TIMESTAMP_OFFSET = 7200;

  private const BATCH_CHUNK_SIZE = 5;

  private const PURGE_OFFSET = 3600;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'language',
    'content_translation',
    'revision_purgatory',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests that cron purges outdated revisions and removes them from history.
   */
  public function testCronPurgesOldRevisions(): void {
    $language_storage = $this->container->get('entity_type.manager')->getStorage('configurable_language');
    $language = $language_storage->load('fr');
    if (!$language) {
      $language = $language_storage->create([
        'id' => 'fr',
        'label' => 'French',
      ]);
      $language->save();
    }

    $this->drupalCreateContentType(['type' => 'article']);
    \Drupal::service('content_translation.manager')->setEnabled('node', 'article', true);

    $node = $this->drupalCreateNode([
      'type' => 'article',
      'title' => 'Original title',
      'status' => self::PUBLISHED_STATUS,
    ]);
    $old_vid = $node->getRevisionId();

    // Create a second revision that will remain after purge.
    $node->setNewRevision(true);
    $node->setTitle('Updated title');
    $node->save();
    $latest_vid = $node->getRevisionId();

    $french = $node->addTranslation('fr', [
      'title' => 'Titre original',
      'status' => self::PUBLISHED_STATUS,
    ]);
    $french->save();
    $translation_old_vid = $french->getRevisionId();

    $node_translation = $node->getTranslation('fr');
    $node_translation->setNewRevision(true);
    $node_translation->setTitle('Titre mis à jour');
    $node_translation->save();

    $connection = $this->container->get('database');
    $time_service = $this->container->get('datetime.time');
    $past_timestamp = $time_service->getRequestTime() - self::PAST_TIMESTAMP_OFFSET;
    $formatted_past = $this->container->get('date.formatter')->format($past_timestamp, 'custom', 'Y-m-d H:i:s');

    // Debugging output. Check in the sites/simpletest/browser-output folder.
    // $this->htmlOutput(sprintf('Backdating legacy revisions to %s.', $formatted_past));
    // $this->htmlOutput($this->getSession()->getPage()->getContent());

    $connection->update('node_revision')
      ->fields(['revision_timestamp' => $past_timestamp])
      ->condition('vid', $old_vid)
      ->execute();
    $connection->update('node_field_revision')
      ->fields(['changed' => $past_timestamp])
      ->condition('vid', $old_vid)
      ->execute();
    $connection->update('node_revision')
      ->fields(['revision_timestamp' => $past_timestamp])
      ->condition('vid', $translation_old_vid)
      ->execute();
    $connection->update('node_field_revision')
      ->fields(['changed' => $past_timestamp])
      ->condition('vid', $translation_old_vid)
      ->execute();

    $this->config('revision_purgatory.settings')
      ->set('enable_auto_purge', true)
      ->set('purge_content_types', [])
      ->set('batch_chunk_size', self::BATCH_CHUNK_SIZE)
      ->set('purge_older_than', $time_service->getRequestTime() - self::PURGE_OFFSET)
      ->save();

    $this->container->get('cron')->run();

    $remaining_vids = $this->container->get('entity_type.manager')->getStorage('node')->revisionIds($node);
    $this->assertNotContains($old_vid, $remaining_vids, 'Legacy base revision ID is no longer present.');
    $this->assertNotContains($translation_old_vid, $remaining_vids, 'Legacy translation revision ID is no longer present.');
    $this->assertNotEmpty($remaining_vids, 'Latest revisions remain after purge.');

    $this->assertFalse($this->revisionExists((int) $old_vid), 'Old revision record removed from storage.');
    $this->assertFalse($this->revisionExists((int) $translation_old_vid), 'Old translation revision record removed from storage.');
  }

  /**
   * Checks whether a revision record exists in the database.
   */
  protected function revisionExists(int $vid): bool {
    $connection = $this->container->get('database');
    $revision_exists = (bool) $connection->select('node_revision', 'nr')
      ->condition('vid', $vid)
      ->countQuery()
      ->execute()
      ->fetchField();

    $field_revision_exists = (bool) $connection->select('node_field_revision', 'nfr')
      ->condition('vid', $vid)
      ->countQuery()
      ->execute()
      ->fetchField();

    return $revision_exists || $field_revision_exists;
  }

}
