<?php

declare(strict_types=1);

namespace Drupal\revision_purgatory\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\ContentEntityStorageInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\revision_purgatory\Form\NodeRevisionContentTypeFilterForm;
use Drupal\revision_purgatory\Services\RevisionPurgatoryRevisionService;
use Drupal\revision_purgatory\Services\UrlNormalizerService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Provides controllers for Revision Purgatory node revision pages.
 */
class NodeRevisionListController extends ControllerBase {

  /**
   * Maximum number of nodes shown per page on the node listing.
   */
  private const NODE_LIST_PAGE_SIZE = 20;

  /**
   * Entity type manager service used via DI to avoid typed override.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManagerService;

  /**
   * Entity field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * Entity repository for handling translated entities.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected EntityRepositoryInterface $entityRepository;

  /**
   * Revision service wrapper.
   *
   * @var \Drupal\revision_purgatory\Services\RevisionPurgatoryRevisionService
   */
  protected RevisionPurgatoryRevisionService $revisionService;

  /**
   * Form builder used for the filter form.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected FormBuilderInterface $filterFormBuilder;

  /**
   * Request stack for retrieving query parameters.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * URL normalizer.
   *
   * @var \Drupal\revision_purgatory\Services\UrlNormalizerService
   */
  protected UrlNormalizerService $urlNormalizer;

  /**
   * Node storage used for revision lookups.
   *
   * @var \Drupal\Core\Entity\ContentEntityStorageInterface
   */
  protected ContentEntityStorageInterface $storage;

    /**
     * Constructs the controller.
     *
     * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
     *   The entity type manager service.
     * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
     *   The entity field manager service.
     * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
     *   The entity repository service.
     * @param \Drupal\revision_purgatory\Services\RevisionPurgatoryRevisionService $revision_service
     *   The revision helper service.
     * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
     *   The request stack service.
     * @param \Drupal\Core\Form\FormBuilderInterface $filter_form_builder
     *   The filter form builder service.
     * @param UrlNormalizerService $url_normalizer
     */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    EntityFieldManagerInterface $entity_field_manager,
    EntityRepositoryInterface $entity_repository,
    RevisionPurgatoryRevisionService $revision_service,
    RequestStack $request_stack,
    FormBuilderInterface $filter_form_builder,
    UrlNormalizerService $url_normalizer
  ) {
    $this->entityTypeManagerService = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->entityRepository = $entity_repository;
    $this->revisionService = $revision_service;
    $this->requestStack = $request_stack;
    $this->filterFormBuilder = $filter_form_builder;
    $this->urlNormalizer = $url_normalizer;
  }

  /**
   * Creates a controller instance using the service container.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   *
   * @return static
   *   The fully constructed controller.
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('entity.repository'),
      $container->get('revision_purgatory.revision_service'),
      $container->get('request_stack'),
      $container->get('form_builder'),
      $container->get('revision_purgatory.url_normalizer')
    );
  }

  /**
   * Builds a preview page for a specific node revision translation.
   *
   * @param int $nid
   *   The node ID.
   * @param string $langcode
   *   The browsing context language code.
   * @param int $vid
   *   The revision ID to preview.
   * @param string $revlangcode
   *   The language code of the revision translation to display.
   *
   * @return array
   *   A render array for the revision preview page.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   Thrown when the requested revision does not exist.
   */
  public function preview(int $nid, string $langcode, int $vid, string $revlangcode): array {
    $this->storage = $this->entityTypeManagerService->getStorage('node');
    $revision = $this->storage->loadRevision($vid);

    if (!$revision) {
      throw new NotFoundHttpException();
    }

    $revision_for_view = $revision;
    if ($revision->hasTranslation($revlangcode)) {
      $revision_for_view = $revision->getTranslation($revlangcode);
    }

    $view_builder = $this->entityTypeManagerService->getViewBuilder('node');
    // Preview mode shows only content. No embedded functional blocks.
    $revision_for_view->in_preview = true;
    $page = $view_builder->view($revision_for_view, 'full', $revlangcode ?? null);

    $back_query = $this->urlNormalizer->currentQuery();
    $back_to_link = Link::fromTextAndUrl(
      $this->t('← Back to revision list'),
      Url::fromRoute(
        'revision_purgatory.node_revisions',
        [
          'nid' => $nid,
          'langcode' => $langcode,
          'vid' => $vid,
        ],
        ['query' => $back_query]
      )
    )->toRenderable();
    $back_to_link_container = [
      '#type' => 'container',
      '#attributes' => ['class' => ['revision-purgatory-back-to-link']],
      'link' => $back_to_link,
    ];

    $build = [
      'revision_purgatory_back_to_link' => $back_to_link_container,
      'node_preview' => $page,
    ];

    return $build;
  }

  /**
   * Lists revisions for a given node version.
   *
   * @param int $nid
   *   The node ID.
   * @param string $langcode
   *   The context language code.
   * @param int $vid
   *   The version ID owning the revisions.
   *
   * @return array
   *   A render array containing revision rows.
   */
  public function revisionList(int $nid, string $langcode, int $vid): array {
    $revisions = $this->revisionService->getRevisionsByVersion($vid);
    $query_params = $this->urlNormalizer->currentQuery();

    foreach ($revisions as $revision) {
      $revision->link_query = $query_params;
    }

    $build = [
      '#theme' => 'revision_purgatory_node_revisions',
      '#title' => $this->t('Node Revision List'),
      '#backtolink' => Link::fromTextAndUrl(
        $this->t('← Back to Version List'),
        Url::fromRoute(
          'revision_purgatory.node_versions',
          [
            'nid' => $nid,
            'langcode' => $langcode,
          ],
          ['query' => $query_params]
        )
      ),
      '#context_langcode' => $langcode,
      '#revisions' => $revisions,
    ];

    return $build;
  }

  /**
   * Lists nodes that can be purged along with their revision counts.
   *
   * @return array
   *   A render array with node rows and pagination.
   */
  public function nodeList(): array {
    $current_request = $this->requestStack->getCurrentRequest();
    $content_type = $current_request->query->get('content_type', '');
    $title = $current_request->query->get('title', '');
    $query_params = $this->urlNormalizer->currentQuery();

    $this->storage = $this->entityTypeManagerService->getStorage('node');
    $query = $this->storage->getQuery()->accessCheck(false);
    if ($content_type) {
      $query->condition('type', $content_type);
    }

    if ($title) {
      $query->condition('title', '%' . $title . '%', 'LIKE');
    }
    $query->sort('nid', 'ASC');
    $nids = $query->pager(self::NODE_LIST_PAGE_SIZE)->execute();
    $nodes = $this->storage->loadMultiple($nids);

    $data = [];
    foreach ($nodes as $node) {
      $detail_query = $this->urlNormalizer->forDetailLink($query_params);
      $back_query = $this->urlNormalizer->forBackLink($query_params);
      $data[] = [
        'title' => $node->getTitle(),
        'nid' => $node->id(),
        'revisions' => $this->revisionService->getRevisionsPerLanguagesBy((int) $node->id()),
        'tables' => [],
        'detail_query' => $detail_query,
        'list_query' => $back_query,
      ];
    }

    return [
      '#filter_form' => $this->filterFormBuilder->getForm(NodeRevisionContentTypeFilterForm::class),
      '#theme' => 'revision_purgatory_nodes',
      '#nodes' => $data,
      '#query' => $back_query,
      '#pager' => ['#type' => 'pager'],
    ];
  }
}
