<?php

declare(strict_types=1);

namespace Drupal\revision_purgatory\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines the settings form for Revision Purgatory.
 */
class RevisionPurgatorySettingsForm extends ConfigFormBase {
  private const DEFAULT_AUTO_PURGE_STATE = 0;

  private const DEFAULT_BATCH_CHUNK_SIZE = 5;

  private const MIN_BATCH_SIZE = 1;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs the settings form.
   */
  public function __construct(ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($config_factory);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * Returns the settings form ID.
   *
   * @return string
   *   The unique form identifier.
   */
  public function getFormId(): string {
    return 'revision_purgatory_settings_form';
  }

  /**
   * Lists configurable keys for this form.
   *
   * @return string[]
   *   An array of editable configuration names.
   */
  protected function getEditableConfigNames(): array {
    return ['revision_purgatory.settings'];
  }

  /**
   * Builds the Revision Purgatory settings form.
   *
   * @param array $form
   *   The base form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   *
   * @return array
   *   The populated form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('revision_purgatory.settings');

    // Field group for automatic purge settings.
    $form['auto_purge_group'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Automatic revision purge'),
    ];

    $form['auto_purge_group']['enable_auto_purge'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable automatic revision purge'),
      '#default_value' => $config->get('enable_auto_purge') ?: self::DEFAULT_AUTO_PURGE_STATE,
    ];

    $form['auto_purge_group']['batch_chunk_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Batch chunk size'),
      '#default_value' => $config->get('batch_chunk_size') ?: self::DEFAULT_BATCH_CHUNK_SIZE,
      '#description' => $this->t('Number of items to process per batch operation.'),
      '#min' => self::MIN_BATCH_SIZE,
      '#states' => [
        'visible' => [
          ':input[name="enable_auto_purge"]' => ['checked' => true],
        ],
      ],
    ];

    // Get all content types.
    $content_types = $this->entityTypeManager->getStorage('node_type')->loadMultiple();
    $type_options = [];
    foreach ($content_types as $type) {
      $type_options[$type->id()] = $type->label();
    }

    $form['auto_purge_group']['purge_content_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Content types to purge'),
      '#options' => $type_options,
      '#default_value' => $config->get('purge_content_types') ?: [],
      '#states' => [
        'visible' => [
          ':input[name="enable_auto_purge"]' => ['checked' => true],
        ],
      ],
    ];

    $form['auto_purge_group']['purge_older_than'] = [
      '#type' => 'date',
      '#title' => $this->t('Remove revisions older than'),
      '#default_value' => $config->get('purge_older_than') ? date('Y-m-d', $config->get('purge_older_than')) : '',
      '#description' => $this->t('Only revisions older than this date will be removed.'),
      '#required' => true,
      '#states' => [
        'visible' => [
          ':input[name="enable_auto_purge"]' => ['checked' => true],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Handles submission of the settings form.
   *
   * @param array $form
   *   The submitted form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The submitted form state containing values.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $auto_purge = $form_state->getValue('enable_auto_purge');
    $purge_content_types = $form_state->getValue('purge_content_types');
    $purge_older_than = $form_state->getValue('purge_older_than');
    $purge_older_than_ts = '';
    if (!empty($purge_older_than)) {
      $purge_older_than_ts = strtotime($purge_older_than . ' 00:00:00');
    }
    $this->config('revision_purgatory.settings')
      ->set('batch_chunk_size', $form_state->getValue('batch_chunk_size'))
      ->set('enable_auto_purge', $auto_purge)
      ->set('purge_content_types', $purge_content_types)
      ->set('purge_older_than', $purge_older_than_ts)
      ->save();

    parent::submitForm($form, $form_state);
  }
}
