<?php

declare(strict_types=1);

namespace Drupal\Tests\revision_purgatory\Unit;

use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\StatementInterface;
use Drupal\Core\Entity\ContentEntityStorageInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueInterface;
use Drupal\Core\Database\Query\SelectInterface;
use Drupal\revision_purgatory\Services\RevisionPurgatoryRevisionService;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * @coversDefaultClass \Drupal\revision_purgatory\Services\RevisionPurgatoryRevisionService
 *
 * @group revision_purgatory
 */
class RevisionPurgatoryRevisionServiceTest extends UnitTestCase {
  private const PURGE_INTERVAL = 3600;

  private const BATCH_CHUNK_SIZE = 5;

  private const REVISION_NODE_ID = 42;

  private const REVISION_ID = 7;

  private const REVISION_TIMESTAMP_OFFSET = 100;

  private const EXPECTED_RUN_SUCCESS = 1;

  private const EXPECTED_RUN_DISABLED = 0;

  /**
   * Provides a fully mocked service instance.
   *
   * @param \Drupal\Core\Config\Config $config
   *   The configuration stub.
   * @param array $selects
   *   Optional select query mocks keyed by "table:alias".
   * @param \Drupal\Core\Queue\QueueInterface|null $queue
   *   Optional queue mock.
   *
   * @return \Drupal\revision_purgatory\Services\RevisionPurgatoryRevisionService
   *   The revision service under test.
   */
  protected function buildService(Config $config, array $selects = [], ?QueueInterface $queue = null): RevisionPurgatoryRevisionService {
    $entity_type_manager = $this->createMock(EntityTypeManagerInterface::class);
    $entity_field_manager = $this->createMock(EntityFieldManagerInterface::class);
    $entity_storage = $this->createMock(ContentEntityStorageInterface::class);
    $entity_type_manager->method('getStorage')
      ->with('node')
      ->willReturn($entity_storage);

    $connection = $this->createMock(Connection::class);
    if ($selects) {
      $connection->method('select')
        ->willReturnCallback(function (string $table, string $alias) use ($selects) {
          $key = sprintf('%s:%s', $table, $alias);
          $this->assertArrayHasKey($key, $selects, sprintf('Unexpected select on %s as %s', $table, $alias));
          return $selects[$key];
        });
    }

    $config_factory = $this->createMock(ConfigFactoryInterface::class);
    $config_factory->method('get')
      ->with('revision_purgatory.settings')
      ->willReturn($config);

    $queue_factory = $this->createMock(QueueFactory::class);
    if ($queue) {
      $queue_factory->expects($this->once())
        ->method('get')
        ->with('revision_purgatory_queue')
        ->willReturn($queue);
    }
    else {
      $queue_factory->expects($this->never())
        ->method('get');
    }

    $language_manager = $this->createMock(LanguageManagerInterface::class);
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getId')->willReturn('en');
    $language_manager->method('getCurrentLanguage')
      ->willReturn($language);

    $container = new ContainerBuilder();
    $container->set('language_manager', $language_manager);
    \Drupal::setContainer($container);

    $logger_factory = $this->createMock(LoggerChannelFactoryInterface::class);
    $logger = $this->createMock(LoggerChannelInterface::class);
    $logger_factory->method('get')->with('revision_purgatory')->willReturn($logger);

    return new RevisionPurgatoryRevisionService(
      $entity_type_manager,
      $connection,
      $config_factory,
      $entity_field_manager,
      $queue_factory,
      $language_manager,
      $logger_factory
    );
  }

  /**
   * Ensures cron run queues revisions when enabled.
   *
   * @covers ::run
   */
  public function testRunQueuesRevisionsWhenEnabled(): void {
    $config = $this->createMock(Config::class);
    $purge_date = time() - self::PURGE_INTERVAL;
    $config->method('get')
      ->willReturnMap([
        ['enable_auto_purge', true],
        ['purge_content_types', []],
        ['purge_older_than', $purge_date],
        ['batch_chunk_size', self::BATCH_CHUNK_SIZE],
      ]);

    $statement = $this->createMock(StatementInterface::class);
    $revision = (object) [
      'nid' => self::REVISION_NODE_ID,
      'vid' => self::REVISION_ID,
      'revision_timestamp' => $purge_date - self::REVISION_TIMESTAMP_OFFSET,
    ];
    $statement->method('fetchAllAssoc')
      ->with('vid')
      ->willReturn([$revision->vid => $revision]);

    $protected_select = $this->createMock(SelectInterface::class);
    $protected_select->method('addField')->willReturnSelf();
    $protected_select->method('condition')->willReturnSelf();
    $protected_select->method('where')->willReturnSelf();

    $newer_select = $this->createMock(SelectInterface::class);
    $newer_select->method('addField')->willReturnSelf();
    $newer_select->method('condition')->willReturnSelf();
    $newer_select->method('where')->willReturnSelf();

    $protected_select->expects($this->once())
      ->method('notExists')
      ->with($newer_select)
      ->willReturnSelf();

    $select = $this->createMock(SelectInterface::class);
    $select->method('fields')->willReturnSelf();
    $select->method('join')->willReturnSelf();
    $select->method('condition')->willReturnSelf();
    $select->method('range')->willReturnSelf();
    $select->method('execute')->willReturn($statement);
    $select->expects($this->once())
      ->method('notExists')
      ->with($protected_select)
      ->willReturnSelf();

    $queue = $this->createMock(QueueInterface::class);
    $queue->expects($this->once())
      ->method('createItem')
      ->with([
        'nid' => self::REVISION_NODE_ID,
        'vid' => self::REVISION_ID,
        'revision_timestamp' => $revision->revision_timestamp,
      ]);

    $service = $this->buildService($config, [
      'node_field_revision:nfr_protected' => $protected_select,
      'node_field_revision:nfr_newer' => $newer_select,
      'node_revision:nr' => $select,
    ], $queue);
    $this->assertSame(self::EXPECTED_RUN_SUCCESS, $service->run());
  }

  /**
   * Ensures cron run exits early when auto purge disabled.
   *
   * @covers ::run
   */
  public function testRunReturnsZeroWhenDisabled(): void {
    $config = $this->createMock(Config::class);
    $config->method('get')
      ->willReturnMap([
        ['enable_auto_purge', false],
      ]);

    $service = $this->buildService($config);
    $this->assertSame(self::EXPECTED_RUN_DISABLED, $service->run());
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    \Drupal::unsetContainer();
    parent::tearDown();
  }

}
