<?php

namespace Drupal\revision_summary;

use Drupal\Component\Diff\Diff;
use Drupal\Core\Entity\ContentEntityInterface;

class CompareRevisions {
 
  /**
   * List fields that have changed in an entity from one of its prior versions.
   *
   * Yes it takes an entity object and a version_id string, if you are testing
   * on the command line or developing where you have an entity ID, like a nid,
   * $entity = \Drupal\node\Entity\Node::load($nid);
   */
  public function listChangedFields(ContentEntityInterface $entity, int $version_id, array $watched_fields = []) {

    $fields = [];

    // https://app-housingworks-net.ddev.site/node/746618/revisions/1461634/view
    // $version_id = 1461634;
    $revision = \Drupal::entityTypeManager()->getStorage('node')->loadRevision($version_id);
    // $entity->getLoadedRevisionId()

    if (!$revision) {
      // @TODO Give a message about an invalid version ID?
      return $fields;
    }

    $diff = \Drupal::service('diff.entity_comparison')->compareRevisions($entity, $revision);

    // Really do not like the format diff provides back, and that it includes
    // fields that are the same, but this is still likely to be easier than
    // writing our own.  Each field is identified as "1234:node.field_example"
    $offset = strlen($entity->id() . ':node.');
    foreach ($diff as $field_id => $field_info) {
      $field_name = substr($field_id, $offset);
      // If we are not looking for this field, move on to the next one.
      if ($watched_fields && !in_array($field_name, $watched_fields)) {
        continue;
      }
      // If the field values do not match, move on to the next one.
      if ($field_info['#data']['#left'] === $field_info['#data']['#right']) {
        continue;
      }
      $fields[$field_name] = $field_info['#name'];
    }

    return $fields;
  }

  /**
   * 
   */
  public function listChangesInField(ContentEntityInterface $entity, int $version_id, string $field_name) {
    $changed_values = [];

    // Making it easier for me to play on the command line.
    // $field_name = 'field_hh_cori_flag';
    // $entity = \Drupal\node\Entity\Node::load(415407);
    // $version_id = 1463792;
    // \Drupal::service('revision_summary.compare_revisions')->listChangesInField($entity, $version_id, $field_name);
  
    $revision = \Drupal::entityTypeManager()->getStorage('node')->loadRevision($version_id);

    if (!$revision) {
      // @TODO Give a message about an invalid version ID?
      return $changed_values;
    }

    $diff = \Drupal::service('diff.entity_comparison')->compareRevisions($entity, $revision);
    $prefix = $entity->id() . ':node.';
    $field_id = $prefix . $field_name;

    // @TODO improve performance by, rather than comparing two entire revisions
    // (for diff is turning every field in both revisions into text, and not
    // even caching as far as i can tell), taking the code (ideally factoring
    // it out in diff) from diff/src/DiffEntityParser.php starting at line 62.
    $field_diff = $diff[$field_id];
    $field_diff_data = $field_diff['#data'];
    // [
    //  "#left" => """
    //    Yes - Possible Misdemeanor Flag\n
    //    Client refused or was not asked
    //    """,
    //  "#right" => "Yes - Possible Felony Flag",
    // ]

    $after = explode("\n", $field_diff_data['#left']);
    $before = explode("\n", $field_diff_data['#right']);

    $changed_values['added'] = array_diff($after, $before);
    $changed_values['removed'] = array_diff($before, $after);

    return $changed_values;
  }

  /**
   * Output added and removed values for the given field as renderable markup.
   */
  public function listChangesInFieldAsMarkup(ContentEntityInterface $entity, int $version_id, string $field_name) {
    $build = [];
    $changes = $this->listChangesInField($entity, $version_id, $field_name);
    // @TODO add optional integrations with InOtherWords module.
    if ($changes['added']) {
      $build['added'] = [
        '#markup' => '<p class="added">' . t('Added:') . ' ' . implode(', ', array_values($changes['added'])) . '</p>',
      ];
    }
    if ($changes['removed']) {
      $build['removed'] = [
        '#markup' => '<p class="removed">' . t('Removed:') . ' ' . implode(', ', array_values($changes['removed'])) . '</p>',
      ];
    }
    return $build;
  }

    /**
   * Output field name with inline added and removed values as renderable markup.
   */
  public function giveFieldNameWithChangesInlineAsMarkup(ContentEntityInterface $entity, int $version_id, string $field_name, $field_label = NULL) {
    $build = [];
    $field_label = $field_label ?: $entity->$field_name->getFieldDefinition()->label();
;
    $changes = $this->listChangesInField($entity, $version_id, $field_name);
    if ($changes['added'] || $changes['removed']) {
      $build['open'] = [
        '#markup' => '<p class="inline"><strong> ' . $field_label . '</strong> ',
      ];
    }
    // @TODO add optional integrations with InOtherWords module.
    if ($changes['added']) {
      $build['added'] = [
        '#markup' => '<span class="added">' . t('added') . ' <em>' . implode(', ', array_values($changes['added'])) . '</em></span>',
      ];
    }
    if ($changes['added'] && $changes['removed']) {
      $build['separator'] = ['#markup' => '; '];
    }
    if ($changes['removed']) {
      $build['removed'] = [
        '#markup' => '<span class="removed">' . t('removed') . ' <em>' . implode(', ', array_values($changes['removed'])) . '</em></span>',
      ];
    }
    if ($changes['added'] || $changes['removed']) {
      $build['close'] = ['#markup' => '</p>'];
    }
    return $build;
  }

  public function latestRevisionIdWithChangedField(ContentEntityInterface $entity, string $field_name, ?bool $fallback_to_first = FALSE) {
    $current_value = $entity->$field_name->value;
    if ($current_value === NULL) {
      if (!$fallback_to_first) {
        return;
      }
      $entity_revision_table = $entity->getEntityTypeId() . '_revision';
      // @TODO make this work for more than nodes; we have to use 'nid' where
      // it ought to be entity_id and 'vid' where it ought to be revision_id.
      $sql = "SELECT vid FROM {$entity_revision_table} WHERE nid=:entity_id ORDER BY vid ASC LIMIT 1";
      return \Drupal::database()->query($sql,
      [
        ':entity_id' => $entity->id(),
      ])->fetchField();
    }
    // SELECT revision_id FROM node_revision__field_date_stamp WHERE entity_id=746618 AND field_date_stamp_value<>'2025-04-05' ORDER BY revision_id DESC LIMIT 1;
    // \Drupal::database()->query("SELECT revision_id FROM :field_revision_table WHERE entity_id=:entity_id AND :field_value_column <> :current_value ORDER BY revision_id DESC LIMIT 1",
    // Drupal does not allow replacing table and field names in queries:
    // "Placeholders cannot be used for column and table names. Instead, if
    // these are derived from unsafe input, they should be run through
    // $database->escapeTable()."
    // per https://www.drupal.org/docs/drupal-apis/database-api/static-queries
    // As the field name is forced to be a string and this is a developer-only
    // method, we will not figure out the escape table thing.
    $field_revision_table = 'node_revision__' . $field_name;
    $field_value_column = $field_name . '_value';
    $sql = "SELECT revision_id FROM {$field_revision_table} WHERE entity_id=:entity_id AND $field_value_column <> :current_value ORDER BY revision_id DESC LIMIT 1";
    return \Drupal::database()->query($sql,
    [
      ':entity_id' => $entity->id(),
      ':current_value' => $current_value,
    ])->fetchField();
  }

  public function oldestRevisionIdWithUnchangedField(ContentEntityInterface $entity, string $field_name) {
  }

}