<?php

declare (strict_types=1);
namespace Drupal\rfc9557\Plugin\DataType;

use DateTimeZone;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\Attribute\DataType;
use Drupal\Core\TypedData\Plugin\DataType\StringData;
use Drupal\Core\TypedData\Type\DateTimeInterface;
use Drupal\rfc9557\DateTimeParts;
use Drupal\rfc9557\ExtendedDateTimeParts;
use Drupal\rfc9557\Plugin\Validation\Constraint\Rfc9557;
use Drupal\rfc9557\TimeZoneParts;

/**
 * Date time typed data for RFC9557 extended dates.
 */
#[DataType(
  id: "datetime_rfc9557",
  label: new TranslatableMarkup("DateTime RFC 9557"),
  description: new TranslatableMarkup("Datetime with optional time zone."),
  constraints: [
    'Rfc9557' => ['mode' => Rfc9557::MODE_STRICT],
  ],
)]
class DateTimeRfc9557 extends StringData implements DateTimeInterface {

  /**
   * {@inheritdoc}
   */
  public function getDateTime(): ?DrupalDateTime {
    if (!$this->value) {
      return NULL;
    }

    $parts = ExtendedDateTimeParts::createFromDateString($this->value);
    if (is_null($parts) || is_null($parts->datetime)) {
      return NULL;
    }

    $datetime = new DrupalDateTime($parts->datetime->toString());
    if (!is_null($parts->timezone)) {
      $datetime->setTimezone(new \DateTimeZone($parts->timezone->value));
    }
    return $datetime;
  }

  /**
   * {@inheritdoc}
   */
  public function setDateTime(DrupalDateTime $datetime, $notify = TRUE): void {
    if ($this->value) {
      // If updating an existing value, maintain format as much as makes sense.
      $original = ExtendedDateTimeParts::createFromDateString($this->value);
      $parts = clone $original;
      if ($original->datetime->offsetIsZ()) {
        $datetime_utc = clone $datetime;
        $datetime_utc->setTimezone(new DateTimeZone('UTC'));
        // Replacement p will treat '-00:00' but also '+00:00' as Z so can't be
        // used more generally.
        // https://www.rfc-editor.org/rfc/rfc9557#name-update-to-rfc-3339
        $parts->datetime = DateTimeParts::createFromDateString($datetime_utc->format('Y-m-d\TH:i:sp'));
      }
      else {
        $parts->datetime = DateTimeParts::createFromDateString($datetime->format('c'));
      }
    }
    else {
      $parts = ExtendedDateTimeParts::createFromDateString($datetime->format('c'));
    }
    if (in_array($datetime->getTimezone()->getName(), DateTimeZone::listIdentifiers())) {
      $timezone = $datetime->getTimezone()->getName();
      $parts->timezone = new TimeZoneParts($timezone);
    }
    $this->setValue($parts->toString(), $notify);
  }

}
