<?php

declare (strict_types=1);
namespace Drupal\rfc9557;

/**
 * Methods to assist validating rfc9557 data.
 */
final class ValidateRfc9557 {

  /**
   * Validate DateTimeParts.
   *
   * @param \Drupal\Rfc9557\DateTimeParts $date
   *   Datetime to be validated.
   *
   * @return bool
   *   True if it is valid.
   */
  public static function validateDateTime(DateTimeParts $date, bool $strict = FALSE): bool {
    $format = \DateTime::RFC3339;
    // Can have fractions of seconds.
    if ($date->fracsec !== '') {
      $format = \DateTime::RFC3339_EXTENDED;
    }
    // If not strict allow years outside 0000 - 9999.
    if (!$strict) {
      $format = str_replace('Y', 'X', $format);
    }
    // Allowed to validate without offset.
    if ($date->offset === '') {
      $format = substr($format, 0, -1);
    }
    try {
      $date_parsed = date_parse_from_format($format, $date->toString());
      return empty($date_parsed['warnings']) && empty($date_parsed['errors']);
    }
    catch (\ValueError $e) {
      return FALSE;
    }
  }

  /**
   * Validate a RFC9557 extended information time zone.
   *
   * @param string $timezone
   *   The time zone to validate.
   *
   * @return bool
   *   True if it is valid.
   */
  public static function validateTimeZone(string $timezone): bool {
    // DateTimeZone is much more relaxed about what will create an object.

    // An offset is a valid time zone.
    if (self::isOffset($timezone)) {
      return TRUE;
    }
    return in_array($timezone, timezone_identifiers_list());
  }

  /**
   * Test if a string is an offset.
   *
   * Does not validate if it is valid offset.
   *
   * @param string $timezone
   *
   * @return bool
   */
  public static function isOffset(string $timezone): bool {
    return (bool) preg_match(',^[+\-]\d{2}:\d{2}$,', $timezone);
  }

  /**
   * Check for consistentency of RFC3339 time-offset and RFC9557 time zone.
   *
   * @param DateTimeParts $datetime
   *   Datetime with offset.
   * @param string $timezone
   *   Extended information time zone.
   *
   * @return bool
   *   True if the time-offset and time zone combination are valid.
   */
  public static function validateOffsetTimeZone(DateTimeParts $date, string $timezone): bool {
    if (self::isOffset($timezone)) {
      // If the timezone is also an offset they should be the same.
      return ($date->offset . $date->offset_hour . ':' . $date->offset_minute) === $timezone;
    }

    $offset_datetime_string = $date->padded()->toString();
    $timezone_date = clone $date;
    $timezone_date->offset = $timezone_date->offset_hour = $timezone_date->offset_minute = '';
    $timezone_datetime_string = $timezone_date->padded()->toString();

    $format = \DateTime::RFC3339;
    // Can have fractions of seconds.
    if ($date->fracsec !== '') {
      $format = \DateTime::RFC3339_EXTENDED;
    }

    $offset_datetime = \DateTimeImmutable::createFromFormat($format, $offset_datetime_string);
    $format = substr($format, 0, -1);
    $timezone_datetime = \DateTimeImmutable::createFromFormat($format, $timezone_datetime_string, new \DateTimeZone($timezone));

    if ($offset_datetime->getTimestamp() != $timezone_datetime->getTimestamp()) {
      return FALSE;
    }

    return TRUE;
  }

}
