<?php

declare (strict_types=1);
namespace Drupal\tests\rfc9557\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\rfc9557\DateTimeParts;
use Drupal\rfc9557\ExtendedDateTimeParts;
use Drupal\rfc9557\SuffixParts;
use Drupal\rfc9557\SuffixPartsList;
use Drupal\rfc9557\TimeZoneParts;
use Drupal\rfc9557\ValidateRfc9557;
use PHPUnit\Framework\Attributes\DataProvider;

/**
 * @covers \Drupal\rfc9557\DateTimeParts
 */
class Rfc9557UtilitiesTest extends UnitTestCase {

  #[DataProvider('validDateTimeData')]
  public function testDateTimeParts(
    string $datetime_input,
    array $datetime_array,
    string $datetime_output,
    string $padded_output,
    string $padded_up_output
  ): void {
    $parts = DateTimeParts::createFromDateString($datetime_input);
    $this->assertEquals($datetime_array, $parts->toArray());
    $this->assertEquals($datetime_output, $parts->toString());
    $padded = $parts->padded();
    $this->assertEquals($padded_output, $padded->toString());
    $padded_up = $parts->padded(round_up: TRUE);
    $this->assertEquals($padded_up_output, $padded_up->toString());
  }

  public static function validDateTimeData(): array {
    return [
      [
        '2020-01-01T01:01:01.123+12:30', [
          'year' => '2020',
          'month' => '01',
          'day' => '01',
          'hour' => '01',
          'minute' => '01',
          'second' => '01',
          'offset' => '+',
          'offset_hour' => '12',
          'offset_minute' => '30',
          'fracsec' => '123',
        ],
        '2020-01-01T01:01:01.123+12:30',
        '2020-01-01T01:01:01.123+12:30',
        '2020-01-01T01:01:01.123+12:30',
      ],
      [
        '2020-01-01T01:01:01+00:00', [
          'year' => '2020',
          'month' => '01',
          'day' => '01',
          'hour' => '01',
          'minute' => '01',
          'second' => '01',
          'offset' => '+',
          'offset_hour' => '00',
          'offset_minute' => '00',
          'fracsec' => '',
        ],
        '2020-01-01T01:01:01+00:00',
        '2020-01-01T01:01:01+00:00',
        '2020-01-01T01:01:01+00:00',
      ],
      [
        '2020-01-01T01:01:01-00:00', [
          'year' => '2020',
          'month' => '01',
          'day' => '01',
          'hour' => '01',
          'minute' => '01',
          'second' => '01',
          'offset' => '-',
          'offset_hour' => '00',
          'offset_minute' => '00',
          'fracsec' => '',
        ],
        '2020-01-01T01:01:01Z',
        '2020-01-01T01:01:01Z',
        '2020-01-01T01:01:01Z',
      ],
      [
        '2020-01-01T01:01:01Z', [
          'year' => '2020',
          'month' => '01',
          'day' => '01',
          'hour' => '01',
          'minute' => '01',
          'second' => '01',
          'offset' => '-',
          'offset_hour' => '00',
          'offset_minute' => '00',
          'fracsec' => '',
        ],
        '2020-01-01T01:01:01Z',
        '2020-01-01T01:01:01Z',
        '2020-01-01T01:01:01Z',
      ],
      [
        '2020-01-01T01:01:01', [
          'year' => '2020',
          'month' => '01',
          'day' => '01',
          'hour' => '01',
          'minute' => '01',
          'second' => '01',
          'offset' => '',
          'offset_hour' => '',
          'offset_minute' => '',
          'fracsec' => '',
        ],
        '2020-01-01T01:01:01',
        '2020-01-01T01:01:01',
        '2020-01-01T01:01:01',
      ],
      [
        '-2020-01-01T01:01', [
          'year' => '-2020',
          'month' => '01',
          'day' => '01',
          'hour' => '01',
          'minute' => '01',
          'second' => '',
          'offset' => '',
          'offset_hour' => '',
          'offset_minute' => '',
          'fracsec' => '',
        ],
        '-2020-01-01T01:01',
        '-2020-01-01T01:01:00',
        '-2020-01-01T01:01:59',
      ],
      [
        '2020-10-06T01', [
          'year' => '2020',
          'month' => '10',
          'day' => '06',
          'hour' => '01',
          'minute' => '',
          'second' => '',
          'offset' => '',
          'offset_hour' => '',
          'offset_minute' => '',
          'fracsec' => '',
        ],
        '2020-10-06T01',
        '2020-10-06T01:00:00',
        '2020-10-06T01:59:59',
      ],
      [
        '2020-01-31', [
          'year' => '2020',
          'month' => '01',
          'day' => '31',
          'hour' => '',
          'minute' => '',
          'second' => '',
          'offset' => '',
          'offset_hour' => '',
          'offset_minute' => '',
          'fracsec' => '',
        ],
        '2020-01-31',
        '2020-01-31T00:00:00',
        '2020-01-31T23:59:59',
      ],
      [
        '9999-12', [
          'year' => '9999',
          'month' => '12',
          'day' => '',
          'hour' => '',
          'minute' => '',
          'second' => '',
          'offset' => '',
          'offset_hour' => '',
          'offset_minute' => '',
          'fracsec' => '',
        ],
        '9999-12',
        '9999-12-01T00:00:00',
        '9999-12-31T23:59:59',
      ],
      [
        '1234', [
          'year' => '1234',
          'month' => '',
          'day' => '',
          'hour' => '',
          'minute' => '',
          'second' => '',
          'offset' => '',
          'offset_hour' => '',
          'offset_minute' => '',
          'fracsec' => '',
        ],
        '1234',
        '1234-01-01T00:00:00',
        '1234-12-31T23:59:59',
      ],
      [
        '2000-11', [
          'year' => '2000',
          'month' => '11',
          'day' => '',
          'hour' => '',
          'minute' => '',
          'second' => '',
          'offset' => '',
          'offset_hour' => '',
          'offset_minute' => '',
          'fracsec' => '',
        ],
        '2000-11',
        '2000-11-01T00:00:00',
        '2000-11-30T23:59:59',
      ],
      [
        '2000-02', [
          'year' => '2000',
          'month' => '02',
          'day' => '',
          'hour' => '',
          'minute' => '',
          'second' => '',
          'offset' => '',
          'offset_hour' => '',
          'offset_minute' => '',
          'fracsec' => '',
        ],
        '2000-02',
        '2000-02-01T00:00:00',
        '2000-02-29T23:59:59',
      ],
    ];
  }

  #[DataProvider('validExtendedDateTimeData')]
  public function testExtendedDateTimeParts(
    string $datetime_input,
    ?DateTimeParts $datetime,
    ?TimeZoneParts $timezone,
    ?SuffixPartsList $suffix
  ): void {
    $parts = ExtendedDateTimeParts::createFromDateString($datetime_input);
    $this->assertEquals($datetime, $parts->datetime);
    $this->assertEquals($timezone, $parts->timezone);
    $this->assertEquals($suffix, $parts->suffix);
    $this->assertEquals($datetime_input, $parts->toString());
  }

  public static function validExtendedDateTimeData(): array {
    return [
      [
        '2020-01-01T01:01:01',
        new DateTimeParts(
          year: '2020',
          month: '01',
          day: '01',
          hour: '01',
          minute: '01',
          second: '01',
        ),
        NULL,
        NULL,
      ],
      [
        '2020-01-01T01:01:01.123',
        new DateTimeParts(
          year: '2020',
          month: '01',
          day: '01',
          hour: '01',
          minute: '01',
          second: '01',
          fracsec: '123',
        ),
        NULL,
        NULL,
      ],
      [
        '2020-01-01T01:01:01.123+12:30',
        new DateTimeParts(
          year: '2020',
          month: '01',
          day: '01',
          hour: '01',
          minute: '01',
          second: '01',
          fracsec: '123',
          offset: '+',
          offset_hour: '12',
          offset_minute: '30',
        ),
        NULL,
        NULL,
      ],
      [
        '2020-01-01T01:01:01.123+12:30[+12:30]',
        new DateTimeParts(
          year: '2020',
          month: '01',
          day: '01',
          hour: '01',
          minute: '01',
          second: '01',
          fracsec: '123',
          offset: '+',
          offset_hour: '12',
          offset_minute: '30',
        ),
        new TimeZoneParts(
          value: '+12:30',
        ),
        NULL,
      ],
      [
        '2020-01-01T01:01:01-11:00[Pacific/Pago_Pago]',
        new DateTimeParts(
          year: '2020',
          month: '01',
          day: '01',
          hour: '01',
          minute: '01',
          second: '01',
          offset: '-',
          offset_hour: '11',
          offset_minute: '00',
        ),
        new TimeZoneParts(
          value: 'Pacific/Pago_Pago',
        ),
        NULL,
      ],
      [
        '2020-01-01T01:01:01Z[!America/Argentina/Catamarca][u-ca=buddhist]',
        new DateTimeParts(
          year: '2020',
          month: '01',
          day: '01',
          hour: '01',
          minute: '01',
          second: '01',
          offset: '-',
          offset_hour: '00',
          offset_minute: '00',
        ),
        new TimeZoneParts(
          value: 'America/Argentina/Catamarca',
          critical: TRUE,
        ),
        new SuffixPartsList(new SuffixParts(key: 'u-ca', value: 'buddhist')),
      ],
      [
        '2020-01-01T01:01:01+00:00[Europe/London][!u-ca=gregory][_other=value][!_bar=baz]',
        new DateTimeParts(
          year: '2020',
          month: '01',
          day: '01',
          hour: '01',
          minute: '01',
          second: '01',
          offset: '+',
          offset_hour: '00',
          offset_minute: '00',
        ),
        new TimeZoneParts(
          value: 'Europe/London',
          critical: FALSE,
        ),
        new SuffixPartsList(
          new SuffixParts(key: 'u-ca', value: 'gregory', critical: TRUE),
          new SuffixParts(key: '_other', value: 'value', critical: FALSE),
          new SuffixParts(key: '_bar', value: 'baz', critical: TRUE),
        ),
      ],
      [
        '2020-01-01T01:01:01Z[u-ca=dangi][!_bar=baz]',
        new DateTimeParts(
          year: '2020',
          month: '01',
          day: '01',
          hour: '01',
          minute: '01',
          second: '01',
          offset: '-',
          offset_hour: '00',
          offset_minute: '00',
        ),
        NULL,
        new SuffixPartsList(
          new SuffixParts(key: 'u-ca', value: 'dangi'),
          new SuffixParts(key: '_bar', value: 'baz', critical: TRUE),
        ),
      ],
    ];
  }

  // Because of the order of writing these validate methods, get more
  // testing in Rfc9557ValidatorTest.

  public function testValidateIsOffset(): void {
    $this->assertTrue(ValidateRfc9557::isOffset('+12:00'));
    $this->assertFalse(ValidateRfc9557::isOffset('Europe/London'));
  }

  public function testValidateTimezone(): void {
    $this->assertTrue(ValidateRfc9557::validateTimeZone('America/North_Dakota/New_Salem'));
    $this->assertTrue(ValidateRfc9557::validateTimeZone('+00:00'));
    $this->assertFalse(ValidateRfc9557::validateTimeZone('Europe/Damascus'));
    $this->assertFalse(ValidateRfc9557::validateTimeZone('14:00'));
  }

  public function testValidateDateTime(): void {
    $date_parts = new DateTimeParts(
      year: '2020',
    );
    $this->assertTrue(ValidateRfc9557::validateDateTime($date_parts->padded()));
    $date_parts = new DateTimeParts(
      year: '102020',
    );
    $this->assertFalse(ValidateRfc9557::validateDateTime($date_parts->padded(), TRUE));
    $date_parts = new DateTimeParts(
      year: '3000',
      offset: '+',
      offset_hour: '01',
      offset_minute: '55',
    );
    $this->assertTrue(ValidateRfc9557::validateDateTime($date_parts->padded()));
  }

  public function testValidateOffsetTimeZone(): void {
    $date_parts = new DateTimeParts(
      year: '2020',
      month: '06',
      day: '15',
      hour: '12',
      minute: '54',
      second: '03',
      offset: '+',
      offset_hour: '09',
      offset_minute: '30',
    );
    $this->assertTrue(ValidateRfc9557::validateOffsetTimeZone($date_parts, 'Australia/Broken_Hill'));
    $date_parts->month = '12';
    $this->assertFalse(ValidateRfc9557::validateOffsetTimeZone($date_parts, 'Australia/Broken_Hill'));
  }

}
