<?php

declare (strict_types=1);
namespace Drupal\rfc9557;

final class ExtendedDateTimeParts {

  public function __construct(
    public DateTimeParts $datetime,
    public ?TimeZoneParts $timezone,
    public ?SuffixPartsList $suffix,
  ) {
  }

  public static function createFromDateString(string $date): ?static {
    $datetime = DateTimeParts::createFromDateString($date);
    if (is_null($datetime)) {
      return NULL;
    }

    $timezone = NULL;
    $suffix = new SuffixPartsList();
    $all_extended = $each_extended = [];
    if (preg_match(
      pattern: '|' . ParseRfc9557::EXTENDED_PATTERN . '+$|',
      subject: $date,
      matches: $all_extended,
    )) {
      preg_match_all(
        pattern: '|' . ParseRfc9557::EXTENDED_PATTERN . '|',
        subject: $all_extended[0],
        matches: $each_extended,
        flags: PREG_SET_ORDER,
      );

      foreach ($each_extended as $extended) {
        // Note that a time-zone is syntactically similar to a suffix-tag
        // but does not include an equals sign. This special case is only
        // available for time zone tags.
        if (!strpos(needle: '=', haystack: $extended[2])) {
          $timezone = new TimeZoneParts(
            value: $extended[2],
            critical: $extended[1] == '!',
          );
        }
        else {
          list($key, $value) = explode(separator: '=', string: $extended[2]);
          $suffix->append(new SuffixParts(
            key: $key,
            value: $value,
            critical: $extended[1] == '!',
          ));
        }
      }
    }

    return new static(
      datetime: $datetime,
      timezone: $timezone,
      suffix: count($suffix) ? $suffix : NULL,
    );
  }

  public function toString(): string {
    return $this->datetime->toString()
      . $this->timezone?->toString()
      . $this->suffix?->toString();
  }

}
