<?php

declare (strict_types=1);
namespace Drupal\tests\rfc9557\Kernel;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\KernelTests\KernelTestBase;
use Drupal\rfc9557\Plugin\DataType\DateTimeRfc9557;
use Drupal\rfc9557\Plugin\Validation\Constraint\Rfc9557;

class DataTypeTest extends KernelTestBase {

  /**
   * The typed data manager to use.
   *
   * @var \Drupal\Core\TypedData\TypedDataManager
   */
  protected $typedDataManager;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'rfc9557',
  ];

  /**
    * {@inheritdoc}
    */
  protected function setUp() : void {
    parent::setUp();
    $this->typedDataManager = $this->container
      ->get('typed_data_manager');
  }

  public function testValue(): void {
    // Create with RFC3339 offset.
    $data_definition = DataDefinition::create('datetime_rfc9557');
    $typed_data = $this->typedDataManager->create($data_definition, '2020-01-01T01:01:01+00:00');
    $this->assertInstanceOf(DateTimeRfc9557::class, $typed_data);
    $this->assertEquals('2020-01-01T01:01:01+00:00', $typed_data->getValue());
    $this->assertEquals(0, $typed_data->validate()->count());
    $this->assertEquals('2020-01-01T01:01:01+00:00', $typed_data->getDateTime()->format('c'));
    $this->assertEquals('+00:00', $typed_data->getDateTime()->getTimezone()->getName());

    // Create with extended RFC9557 with Z.
    $typed_data = $this->typedDataManager->create($data_definition, '2020-01-01T01:01:01Z[Europe/Berlin]');
    $this->assertInstanceOf(DateTimeRfc9557::class, $typed_data);
    $this->assertEquals('2020-01-01T01:01:01Z[Europe/Berlin]', $typed_data->getValue());
    $this->assertEquals(0, $typed_data->validate()->count());
    $this->assertEquals('2020-01-01T02:01:01+01:00', $typed_data->getDateTime()->format('c'));
    $this->assertEquals('Europe/Berlin', $typed_data->getDateTime()->getTimezone()->getName());

    // Create with extended RFC9557 with offset.
    $typed_data = $this->typedDataManager->create($data_definition, '2020-01-01T01:01:01+01:00[Europe/Berlin]');
    $this->assertInstanceOf(DateTimeRfc9557::class, $typed_data);
    $this->assertEquals('2020-01-01T01:01:01+01:00[Europe/Berlin]', $typed_data->getValue());
    $this->assertEquals(0, $typed_data->validate()->count());
    $this->assertEquals('2020-01-01T01:01:01+01:00', $typed_data->getDateTime()->format('c'));
    $this->assertEquals('Europe/Berlin', $typed_data->getDateTime()->getTimezone()->getName());

    // Check validation is run.
    $typed_data = $this->typedDataManager->create($data_definition, '2020-01-01T01:01:01Z[Solar/Moon]');
    $this->assertInstanceOf(DateTimeRfc9557::class, $typed_data);
    $this->assertEquals(2, $typed_data->validate()->count());
    // PrimativeTypeConstraint.
    $this->assertEquals('This value should be of the correct primitive type.', $typed_data->validate()->get(0)->getMessage());
    // Rfc9557.
    $this->assertEquals(Rfc9557::INVALID_TIMEZONE, $typed_data->validate()->get(1)->getCode());

    // Check ::setDateTime() with offset.
    $typed_data = $this->typedDataManager->create($data_definition);
    $this->assertInstanceOf(DateTimeRfc9557::class, $typed_data);
    $this->assertEquals(0, $typed_data->validate()->count());
    $this->assertNull($typed_data->getValue());
    $date_time = new DrupalDateTime('2020-01-01T01:01:01+00:00');
    $typed_data->setDateTime($date_time);
    $this->assertEquals('2020-01-01T01:01:01+00:00', $typed_data->getDateTime()->format('c'));
    $this->assertEquals('+00:00', $typed_data->getDateTime()->getTimezone()->getName());
    $this->assertEquals('2020-01-01T01:01:01+00:00', $typed_data->getValue());

    // With location timezone.
    $typed_data = $this->typedDataManager->create($data_definition);
    $this->assertInstanceOf(DateTimeRfc9557::class, $typed_data);
    $date_time = new DrupalDateTime('2020-01-01T01:01:01', new \DateTimeZone('Europe/Berlin'));
    $typed_data->setDateTime($date_time);
    $this->assertEquals('2020-01-01T01:01:01+01:00', $typed_data->getDateTime()->format('c'));
    $this->assertEquals('Europe/Berlin', $typed_data->getDateTime()->getTimezone()->getName());
    $this->assertEquals('2020-01-01T01:01:01+01:00[Europe/Berlin]', $typed_data->getValue());

    // Maintains uneffected values when updating time.
    $typed_data = $this->typedDataManager->create($data_definition, '2020-01-01T01:01:01Z[Europe/Berlin][u-ca=hebrew][_foo=bar]');
    $this->assertInstanceOf(DateTimeRfc9557::class, $typed_data);
    // Using 'UTC' here would be a timezone rather than ofset and change the
    // [timezone]. '-00:00' is equivalent to 'Z' in RFC9557
    // https://www.rfc-editor.org/rfc/rfc9557#name-update-to-rfc-3339.
    $date_time = new DrupalDateTime('2022-01-01T01:01:01', new \DateTimeZone('-00:00'));
    $typed_data->setDateTime($date_time);
    $this->assertEquals('2022-01-01T02:01:01+01:00', $typed_data->getDateTime()->format('c'));
    $this->assertEquals('Europe/Berlin', $typed_data->getDateTime()->getTimezone()->getName());
    $this->assertEquals('2022-01-01T01:01:01Z[Europe/Berlin][u-ca=hebrew][_foo=bar]', $typed_data->getValue());
    $date_time = new DrupalDateTime('2022-01-01T01:01:01', new \DateTimeZone('Europe/Amsterdam'));
    $typed_data->setDateTime($date_time);
    $this->assertEquals('2022-01-01T01:01:01+01:00', $typed_data->getDateTime()->format('c'));
    $this->assertEquals('Europe/Amsterdam', $typed_data->getDateTime()->getTimezone()->getName());
    $this->assertEquals('2022-01-01T00:01:01Z[Europe/Amsterdam][u-ca=hebrew][_foo=bar]', $typed_data->getValue());
  }

}
