<?php

declare (strict_types=1);

namespace Drupal\rfc9557\Plugin\Validation\Constraint;

use Drupal\rfc9557\Model\ExtendedDateTimeParts;
use Drupal\rfc9557\ValidateRfc9557;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;
use Symfony\Component\Validator\Exception\UnexpectedTypeException;

/**
 * Validates the Rfc9557 constraint.
 */
class Rfc9557Validator extends ConstraintValidator {

  /**
   * {@inheritdoc}
   */
  public function validate(mixed $value, Constraint $constraint): void {
    if (!$constraint instanceof Rfc9557) {
      throw new UnexpectedTypeException($constraint, __NAMESPACE__ . '\Rfc9557');
    }

    if (NULL === $value) {
      return;
    }

    $parts = ExtendedDateTimeParts::createFromDateString($value);
    if (!ValidateRfc9557::validateDateTime($parts->datetime, $constraint->mode == Rfc9557::MODE_STRICT)) {
      $this->context->buildViolation($constraint->invalidDatetimeMessage)
        ->setCode(Rfc9557::INVALID_DATETIME)
        ->addViolation();

      return;
    }
    if ($constraint->mode == Rfc9557::MODE_STRICT
      && $parts->datetime->offset === ''
    ) {
      $this->context->buildViolation($constraint->invalidDatetimeMessage)
        ->setCode(Rfc9557::INVALID_DATETIME)
        ->addViolation();

      return;
    }
    if (!is_null($parts->timezone) &&
      !ValidateRfc9557::validateTimeZone($parts->timezone->value)
    ) {
      $this->context->buildViolation($constraint->invalidTimeZoneMessage)
        ->setCode(Rfc9557::INVALID_TIMEZONE)
        ->addViolation();

      return;
    }
    // An offset in a extended information time zone needs to be compared.
    // Otherwise if the time zone is marked as critical.
    // Inconsistency can also be resolved.
    // https://datatracker.ietf.org/doc/html/rfc9557#section-3.4
    if (
      !is_null($parts->timezone) &&
      !$parts->datetime->offsetIsZ() &&
      $constraint->mode == Rfc9557::MODE_STRICT &&
      ($parts->timezone->critical || ValidateRfc9557::isOffset($parts->timezone->value)) &&
      !ValidateRfc9557::validateOffsetTimeZone($parts->datetime, $parts->timezone->value)
    ) {
      $this->context->buildViolation($constraint->offsetInconsistentMessage)
        ->setCode(Rfc9557::OFFSET_INCONSISTENT)
        ->addViolation();
      return;
    }
  }

}
