<?php

declare (strict_types=1);

namespace Drupal\rfc9557\Model;

use Drupal\rfc9557\ParseRfc9557;

/**
 * RFC9557 date split into its parts.
 *
 * @see \Drupal\rfc9557\Model\DateTimeParts
 * @see \Drupal\rfc9557\Model\TimeZoneParts
 * @see \Drupal\rfc9557\Model\SuffixPartsList
 */
final class ExtendedDateTimeParts {

  public function __construct(
    public DateTimeParts $datetime,
    public ?TimeZoneParts $timezone,
    public ?SuffixPartsList $suffix,
  ) {
  }

  /**
   * Create parts from a full RFC9557 string.
   */
  public static function createFromDateString(string $date): ?static {
    $datetime = DateTimeParts::createFromDateString($date);
    if (is_null($datetime)) {
      return NULL;
    }

    $timezone = NULL;
    $suffix = new SuffixPartsList();
    $all_extended = $each_extended = [];
    if (preg_match(
      pattern: '|' . ParseRfc9557::EXTENDED_PATTERN . '+$|',
      subject: $date,
      matches: $all_extended,
    )) {
      preg_match_all(
        pattern: '|' . ParseRfc9557::EXTENDED_PATTERN . '|',
        subject: $all_extended[0],
        matches: $each_extended,
        flags: PREG_SET_ORDER,
      );

      foreach ($each_extended as $extended) {
        // Note that a time-zone is syntactically similar to a suffix-tag
        // but does not include an equals sign. This special case is only
        // available for time zone tags.
        if (!strpos(needle: '=', haystack: $extended[2])) {
          $timezone = new TimeZoneParts(
            value: $extended[2],
            critical: $extended[1] == '!',
          );
        }
        else {
          [$key, $value] = explode(separator: '=', string: $extended[2]);
          $suffix->append(new SuffixParts(
            key: $key,
            value: $value,
            critical: $extended[1] == '!',
          ));
        }
      }
    }

    return new static(
      datetime: $datetime,
      timezone: $timezone,
      suffix: count($suffix) ? $suffix : NULL,
    );
  }

  /**
   * Return the parts as a RFC9557 string.
   */
  public function toString(): string {
    return $this->datetime->toString()
      . $this->timezone?->toString()
      . $this->suffix?->toString();
  }

}
