<?php

declare (strict_types=1);

namespace Drupal\rfc9557\Model;

use Drupal\Component\Datetime\DateTimePlus;
use Drupal\rfc9557\ParseRfc9557;

/**
 * Date Time parts and methods to manipulate them.
 *
 * @see \Drupal\rfc9557\Model\ExtendedDateTimeParts
 */
final class DateTimeParts {

  public function __construct(
    public string $year,
    public string $month = '',
    public string $day = '',
    public string $hour = '',
    public string $minute = '',
    public string $second = '',
    public string $fracsec = '',
    public string $offset = '',
    public string $offset_hour = '',
    public string $offset_minute = '',
  ) {
  }

  /**
   * Split a date string into parts.
   *
   * Similar to date_parse but without validation and with different offset
   * handling for later RFC9557 validation.
   *
   * @param string $date
   *   Date in (partial) RFC3339 like format.
   *
   * @return static
   */
  public static function createFromDateString(string $date): ?static {
    $date_matches = $time_matches = $fracsec_matches = $offset_matches = $extended_matches = [];
    if (!preg_match(
      pattern: ParseRfc9557::DATE_PATTERN,
      subject: $date,
      matches: $date_matches
    )) {
      return NULL;
    }
    preg_match(
      pattern: ParseRfc9557::TIME_PATTERN,
      subject: $date,
      matches: $time_matches
    );
    preg_match(
      pattern: ParseRfc9557::FRACSEC_PATTERN,
      subject: $date,
      matches: $fracsec_matches,
    );

    preg_match(
      pattern: ParseRfc9557::OFFSET_PATTERN,
      subject: $date,
      matches: $offset_matches
    );
    if (preg_match(
      pattern: ParseRfc9557::Z_PATTERN,
      subject: $date
    )) {
      $offset_matches = ['', '-', '00', '00'];
    }

    return new static(
      year: $date_matches[1],
      month: $date_matches[2] ?? '',
      day: $date_matches[3] ?? '',
      hour: $time_matches[1] ?? '',
      minute: $time_matches[2] ?? '',
      second: $time_matches[3] ?? '',
      fracsec: $fracsec_matches[1] ?? '',
      offset: $offset_matches[1] ?? '',
      offset_hour: $offset_matches[2] ?? '',
      offset_minute: $offset_matches[3] ?? '',
    );
  }

  /**
   * Get padded full datetime.
   *
   * If this is a reduced accuracy date first values will be defaulted.
   *
   * @param bool $round_up
   *   (Optional) Round to the end of the unknown period.
   *   Default to adding the minimum value.
   *
   * @return static
   *   A new DateTimeParts object with parts completed.
   */
  public function padded(bool $round_up = FALSE): static {
    $padded_date = clone $this;
    $padded_date->month = $padded_date->month ?: ($round_up ? '12' : '01');
    $padded_date->day = $padded_date->day ?: (
      $round_up ?
        (string) $this->calDaysInMonth((int) $padded_date->month, (int) $padded_date->year) :
        '01'
    );
    $padded_date->hour = $padded_date->hour ?: ($round_up ? '23' : '00');
    $padded_date->minute = $padded_date->minute ?: ($round_up ? '59' : '00');
    // Seconds should be safe as 59 == 60 leap.
    $padded_date->second = $padded_date->second ?: ($round_up ? '59' : '00');
    return $padded_date;
  }

  /**
   * Return an ISO8610/RFC3339 like datetime string.
   */
  public function toString(): string {
    // @todo can replace DateTimePlus function here.
    // Decide if leading 0 padding should happen on class creation.
    $datetime = DateTimePlus::arrayToISO($this->toArray());
    if ($this->fracsec !== '') {
      $datetime .= '.' . $this->fracsec;
    }
    if ($this->offset !== '') {
      // Z notation for '-00:00' is preferred.
      // https://www.rfc-editor.org/rfc/rfc9557.html#section-2.3
      if ($this->offsetIsZ()) {
        $datetime .= 'Z';
      }
      else {
        $datetime .= $this->offset . $this->offset_hour . ':' . $this->offset_minute;
      }
    }
    return $datetime;
  }

  /**
   * Helper method to check if offset is Z.
   */
  public function offsetIsZ(): bool {
    return $this->offset === '-' && $this->offset_hour === '00' && $this->offset_minute === '00';
  }

  /**
   * Convert to a Drupal DateTimePlus like keyed array.
   */
  public function toArray(): array {
    return (array) $this;
  }

  /**
   * Fallback handler for cal_days_in_month gregorian.
   */
  private function calDaysInMonth(int $month, int $year): int {
    return function_exists('cal_days_in_month') ?
      cal_days_in_month(\CAL_GREGORIAN, $month, $year) :
      (int) date('t', mktime(0, 0, 0, $month, 1, $year));
  }

}
