<?php

declare (strict_types=1);

namespace Drupal\tests\rfc9557\Unit;

use Drupal\rfc9557\Plugin\Validation\Constraint\Rfc9557;
use Drupal\rfc9557\Plugin\Validation\Constraint\Rfc9557Validator;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\DataProvider;
use Symfony\Component\Validator\Context\ExecutionContextInterface;
use Symfony\Component\Validator\Violation\ConstraintViolationBuilderInterface;

/**
 * @covers \Drupal\rfc9557\Plugin\Validation\Constraint\Rfc9557Validator
 * @group validation
 */
class Rfc9557ValidatorTest extends UnitTestCase {

  /**
   * Test date time part validator with valid data.
   */
  #[DataProvider('validDateTimeData')]
  public function testValid(string $datetime, string $mode): void {
    $context = $this->createMock(ExecutionContextInterface::class);
    $context->expects($this->never())
      ->method('buildViolation');
    $validator = new Rfc9557Validator();
    $validator->initialize($context);
    $constraint = new Rfc9557();
    $constraint->mode = $mode;
    $validator->validate($datetime, $constraint);
  }

  /**
   * Valid date times.
   */
  public static function validDateTimeData(): array {
    return [
      ['2020-01-01T23:59:59', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
      ['2020-01-01T23:59:59-00:00', Rfc9557::MODE_STANDARDS],
      ['2020-01-01T23:59:59-00:00', Rfc9557::MODE_STRICT],
      ['2020-01-01T23:59:59+12:00', Rfc9557::MODE_STANDARDS],
      ['2020-01-01T23:59:59+12:00', Rfc9557::MODE_STRICT],
      ['2020-01-01T23:59:59Z', Rfc9557::MODE_STANDARDS],
      ['2020-01-01T23:59:59Z', Rfc9557::MODE_STRICT],
      ['2020-01-01T23:59:59.023+02:00', Rfc9557::MODE_STANDARDS],
      ['2020-01-01T23:59:59.023+02:00', Rfc9557::MODE_STRICT],
      ['2020-01-01T23:59:59.023Z', Rfc9557::MODE_STANDARDS],
      ['2020-01-01T23:59:59.023Z', Rfc9557::MODE_STRICT],
      ['9999-12-31T23:59:59.023Z', Rfc9557::MODE_STANDARDS],
      ['9999-12-31T23:59:59.023Z', Rfc9557::MODE_STRICT],
      ['999999-12-31T23:59:59.023Z', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
      ['2020-01-01T23:59:59[+03:30]', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
      ['2020-01-01T23:59:59+03:30[+03:30]', Rfc9557::MODE_STANDARDS],
      ['2020-01-01T23:59:59+03:30[+03:30]', Rfc9557::MODE_STRICT],
      ['2020-01-01T23:59:59+02:30[+03:30]', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
      ['2020-01-01T23:59:59Z[Pacific/Apia]', Rfc9557::MODE_STANDARDS],
      ['2020-01-01T23:59:59Z[Pacific/Apia]', Rfc9557::MODE_STRICT],
      ['2020-01-01T23:59:59Z[!Pacific/Apia]', Rfc9557::MODE_STANDARDS],
      ['2020-01-01T23:59:59Z[!Pacific/Apia]', Rfc9557::MODE_STRICT],
      ['2020-01-01T23:59:59-00:00[!Pacific/Apia]', Rfc9557::MODE_STANDARDS],
      ['2020-01-01T23:59:59-00:00[!Pacific/Apia]', Rfc9557::MODE_STRICT],
      ['2020-01-01T23:59:59+00:00[!Pacific/Apia]', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
      ['2020-01-01T23:59:59Z[Pacific/Apia][u-ca=japanese]', Rfc9557::MODE_STANDARDS],
      ['2020-01-01T23:59:59Z[Pacific/Apia][u-ca=japanese][_foo=bar]', Rfc9557::MODE_STRICT],
      ['-2020-01-01T23:59:59', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
      ['-2020-01-01T23:59:59-00:00', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
      ['-2020-01-01T23:59:59+12:00', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
      ['-2020-01-01T23:59:59Z', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
      ['-2020-01-01T23:59:59.023+02:00', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
      ['-2020-01-01T23:59:59.023Z', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
      // Strict has an invalid test.
      ['-2020-01-01T23:59:59[+03:30]', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
      ['-2020-01-01T23:59:59+03:30[+03:30]', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
      ['-2020-01-01T23:59:59+02:30[+03:30]', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
      ['-2020-01-01T23:59:59Z[Pacific/Apia]', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
      ['-2020-01-01T23:59:59Z[Pacific/Apia][u-ca=japanese]', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
      ['-9999-12-31T23:59:59.023Z', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
      ['-999999-12-31T23:59:59.023Z', Rfc9557::MODE_STANDARDS],
      // Strict has an invalid test.
    ];
  }

  /**
   * Test date time validator with invalid data.
   */
  #[DataProvider('invalidDateTimeData')]
  public function testInvalid(string $datetime, string $mode, string $error_code): void {
    $context = $this->createMock(ExecutionContextInterface::class);
    $violation_builder = $this->createMock(ConstraintViolationBuilderInterface::class);
    $context->expects($this->atLeastOnce())
      ->method('buildViolation')
      ->willReturn($violation_builder);
    $violation_builder->expects($this->atLeastOnce())
      ->method('setCode')
      ->with($error_code)
      ->willReturnSelf();
    $violation_builder->expects($this->atLeastOnce())
      ->method('addViolation');
    $validator = new Rfc9557Validator();
    $validator->initialize($context);
    $constraint = new Rfc9557();
    $constraint->mode = $mode;
    $validator->validate($datetime, $constraint);
  }

  /**
   * Invalid date time data.
   */
  public static function invalidDateTimeData(): array {
    return [
      ['2020-01-01', Rfc9557::MODE_STANDARDS, Rfc9557::INVALID_DATETIME],
      ['2020-01-01', Rfc9557::MODE_STRICT, Rfc9557::INVALID_DATETIME],
      ['2020-13-01T01:01:01', Rfc9557::MODE_STANDARDS, Rfc9557::INVALID_DATETIME],
      ['2020-01-01T23:59:59', Rfc9557::MODE_STRICT, Rfc9557::INVALID_DATETIME],
      ['2020-01-01T23:59:59[+03:30]', Rfc9557::MODE_STRICT, Rfc9557::INVALID_DATETIME],
      ['2020-01-01T23:59:59+02:30[+03:30]', Rfc9557::MODE_STRICT, Rfc9557::OFFSET_INCONSISTENT],
      ['2020-01-01T23:59:59+00:00[!Pacific/Apia]', Rfc9557::MODE_STRICT, Rfc9557::OFFSET_INCONSISTENT],
      ['2020-01-01T23:59:59+02:30[+03:30:00]', Rfc9557::MODE_STRICT, Rfc9557::INVALID_TIMEZONE],
      ['999999-12-31T23:59:59.023Z', Rfc9557::MODE_STRICT, Rfc9557::INVALID_DATETIME],
      ['-2020-01-01T23:59:59-00:00', Rfc9557::MODE_STRICT, Rfc9557::INVALID_DATETIME],
      ['-2020-01-01T23:59:59+12:00', Rfc9557::MODE_STRICT, Rfc9557::INVALID_DATETIME],
      ['-2020-01-01T23:59:59Z', Rfc9557::MODE_STRICT, Rfc9557::INVALID_DATETIME],
      ['-2020-01-01T23:59:59.023+02:00', Rfc9557::MODE_STRICT, Rfc9557::INVALID_DATETIME],
      ['-2020-01-01T23:59:59.023Z', Rfc9557::MODE_STRICT, Rfc9557::INVALID_DATETIME],
      ['-9999-12-31T23:59:59.023Z', Rfc9557::MODE_STRICT, Rfc9557::INVALID_DATETIME],
      ['-2020-01-01T23:59:59+03:30[+03:30]', Rfc9557::MODE_STRICT, Rfc9557::INVALID_DATETIME],
      ['-2020-01-01T23:59:59Z[Pacific/Apia]', Rfc9557::MODE_STRICT, Rfc9557::INVALID_DATETIME],
      ['-2020-01-01T23:59:59Z[Pacific/Apia][u-ca=japanese][_foo=bar]', Rfc9557::MODE_STRICT, Rfc9557::INVALID_DATETIME],
      ['-9999-12-31T23:59:59.023Z', Rfc9557::MODE_STRICT, Rfc9557::INVALID_DATETIME],
      ['-999999-12-31T23:59:59.023Z', Rfc9557::MODE_STRICT, Rfc9557::INVALID_DATETIME],
    ];
  }

}
