# Extending RIFT Picture Element Builders

This document explains how contrib modules can extend or override the picture element generation functionality in RIFT.

## Overview

RIFT now uses a plugin system for building picture elements, allowing contrib modules to:

- Add custom attributes to picture elements
- Modify the structure of generated HTML
- Add custom logic based on media entity properties
- Override the default behavior for specific cases

## Plugin System Architecture

The picture element builder system consists of:

1. **RiftBuilderInterface**: Interface that all builders must implement
2. **RiftBuilderManager**: Plugin manager that handles discovery and selection
3. **DefaultBuilder**: Default implementation containing the original logic
4. **Custom Builders**: Contrib module implementations

## Creating a Custom Builder

### 1. Create the Plugin Class

Create a new class in your module's `src/Plugin/RiftBuilder/` directory:

```php
<?php

namespace Drupal\your_module\Plugin\RiftBuilder;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\media\MediaInterface;
use Drupal\rift\Attribute\RiftBuilder;
use Drupal\rift\DTO\PictureConfig;
use Drupal\rift\Html\PictureElement;
use Drupal\rift\Plugin\RiftBuilder\DefaultBuilder;
use Drupal\rift\RiftPicture;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Custom picture element builder for your module.
 */
#[RiftBuilder(
  id: 'your_custom_builder',
  label: new TranslatableMarkup('Your Custom Picture Element Builder'),
  description: new TranslatableMarkup('Custom builder that adds specific functionality.'),
  priority: 10
)]
class YourCustomBuilder extends DefaultBuilder implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static();
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(array $sources, ?MediaInterface $media, PictureConfig $pictureConfig): bool {
    // Define when this builder should be used.
    // Return TRUE if this builder should handle this case.

    // Example: Check if media has a specific field
    if ($media && $media->hasField('field_your_custom_field')) {
      return !$media->get('field_your_custom_field')->isEmpty();
    }

    // Example: Check configuration
    if (isset($pictureConfig->attribute['data-use-custom-builder'])) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function build(array $sources, ?MediaInterface $media, PictureConfig $pictureConfig, RiftPicture $riftPicture): PictureElement {
    // Get the default picture element from parent.
    $picture_element = parent::build($sources, $media, $pictureConfig, $riftPicture);

    // Add your custom modifications here.

    // Example: Add custom attributes
    $current_attributes = $picture_element->getAttribute();
    $current_attributes->setAttribute('data-your-module', 'true');
    $current_attributes->setAttribute('data-media-id', $media ? $media->id() : '');

    // Example: Add custom CSS classes
    $current_classes = $current_attributes->getClass();
    $current_classes->add('your-module-picture');

    // Set the modified attributes back.
    $picture_element->setAttribute($current_attributes);

    // Example: Modify source elements
    $sources = $picture_element->getSource();
    foreach ($sources as $source) {
      $source_attributes = $source->getAttribute();
      $source_attributes->setAttribute('data-custom-source', 'true');
      $source->setAttribute($source_attributes);
    }

    return $picture_element;
  }

  /**
   * {@inheritdoc}
   */
  public function getPriority(): int {
    return 10; // Higher priority than default (0)
  }

}
```

### 2. Required Methods

#### `isApplicable()`
This method determines when your builder should be used. Return `TRUE` if your builder should handle the current case.

#### `build()`
This method builds the picture element. You can:
- Call `parent::build()` to get the default implementation
- Modify the returned picture element
- Or implement completely custom logic

#### `getPriority()`
Return a priority value. Higher priority builders are checked first. Default builder has priority 0.

### 3. Available Helper Methods

The `RiftPicture` service provides these public methods for builders:

- `getMediaSourcePlugin()`: Access the media source plugin
- `generateImageUrlFromStyles()`: Generate image URLs from style arrays

## Examples

### Adding Custom Attributes

```php
public function build(array $sources, ?MediaInterface $media, PictureConfig $pictureConfig, RiftPicture $riftPicture): PictureElement {
  $picture_element = parent::build($sources, $media, $pictureConfig, $riftPicture);

  // Add custom data attributes
  $attributes = $picture_element->getAttribute();
  $attributes->setAttribute('data-lazy-load', 'true');
  $attributes->setAttribute('data-media-type', $media ? $media->bundle() : '');

  $picture_element->setAttribute($attributes);

  return $picture_element;
}
```

### Conditional Modifications

```php
public function isApplicable(array $sources, ?MediaInterface $media, PictureConfig $pictureConfig): bool {
  // Only apply for specific media types
  if ($media && $media->bundle() === 'your_custom_media_type') {
    return TRUE;
  }

  // Or based on user permissions
  if (\Drupal::currentUser()->hasPermission('use custom picture builder')) {
    return TRUE;
  }

  return FALSE;
}
```

### Completely Custom Implementation

```php
public function build(array $sources, ?MediaInterface $media, PictureConfig $pictureConfig, RiftPicture $riftPicture): PictureElement {
  // Don't call parent - implement completely custom logic
  $picture_element = new PictureElement();

  // Your custom implementation here...

  return $picture_element;
}
```

## Plugin Discovery

Your plugin will be automatically discovered if:

1. It's in the correct namespace: `Drupal\your_module\Plugin\RiftBuilder\`
2. It implements `RiftBuilderInterface`
3. It has the `#[RiftBuilder]` attribute
4. Your module is enabled

## Testing Your Builder

You can test your builder by:

1. Creating a media entity that meets your `isApplicable()` criteria
2. Using the `rift_picture` Twig filter
3. Inspecting the generated HTML for your custom attributes/modifications

## Best Practices

1. **Extend DefaultBuilder**: Unless you need completely different behavior, extend `DefaultBuilder` to maintain compatibility
2. **Check Applicability**: Always implement proper `isApplicable()` logic to avoid conflicts
3. **Use Priority Wisely**: Set appropriate priority values to ensure your builder runs when needed
4. **Document Your Changes**: Add clear documentation about what your builder does
5. **Test Thoroughly**: Ensure your builder works with different media types and configurations

## Troubleshooting

### Builder Not Being Used
- Check that `isApplicable()` returns `TRUE` for your test case
- Verify your plugin is in the correct namespace
- Clear Drupal's cache after adding your plugin

### Conflicts with Other Builders
- Adjust your `getPriority()` value
- Make your `isApplicable()` logic more specific
- Check for conflicts with other modules

### Performance Issues
- Keep `isApplicable()` logic lightweight
- Avoid expensive operations in the builder
- Consider caching if needed
