# RIFT UI Config - API Reference

This document provides a comprehensive reference for all APIs, stores, functions, and data structures used in the RIFT UI Config application.

## 📋 Table of Contents

- [Stores](#stores)
- [Data Types](#data-types)
- [Transformation Functions](#transformation-functions)
- [Validation Functions](#validation-functions)
- [Component APIs](#component-apis)
- [Build System](#build-system)

---

## Stores

### **drupalSettings**

**Type**: `Writable<DrupalSettings>`  
**Purpose**: Central store containing all Drupal configuration data

```typescript
export let drupalSettings = writable<DrupalSettings>({
  source: "",
  media_source: "",
  config: {
    multipliers: [],
    screens: [],
    formats: [],
    attributes: [],
    transform: "",
    fallback_transform: ""
  },
  aspect_ratios: [],
  view_modes: {},
  app_config: {
    source: {},
    media_source: {},
    aspect_ratios: {},
    formats: {}
  }
});
```

**Subscriptions**:
- `multipliers.subscribe()`: Updates `drupalSettings.config.multipliers`
- `screens.subscribe()`: Updates `drupalSettings.config.screens`
- `formats.subscribe()`: Updates `drupalSettings.config.formats`
- `aspect_ratios.subscribe()`: Updates `drupalSettings.aspect_ratios`
- `attributes.subscribe()`: Updates `drupalSettings.config.attributes`
- `viewModes.subscribe()`: Updates `drupalSettings.view_modes`

### **aspect_ratios**

**Type**: `Writable<AspectRatio[]>`  
**Purpose**: Stores aspect ratio definitions

```typescript
export let aspect_ratios = writable<AspectRatio[]>([
  { id: 0, value: "16:9" },
  { id: 1, value: "4:3" }
]);
```

### **viewModes**

**Type**: `Writable<ViewMode[]>`  
**Purpose**: Stores view mode configurations

```typescript
export let viewModes = writable<ViewMode[]>([
  {
    id: 0,
    name: "default",
    sizes: [
      { screen: "mobile", width: "100%", aspect_ratio: "16:9", enabled: true }
    ]
  }
]);
```

### **screens**

**Type**: `Writable<Screen[]>`  
**Purpose**: Stores screen breakpoint definitions

```typescript
export let screens = writable<Screen[]>([
  { id: 0, media_query: "(max-width: 768px)", width: "100%" }
]);
```

### **multipliers**

**Type**: `Writable<Multiplier[]>`  
**Purpose**: Stores image multiplier and quality settings

```typescript
export let multipliers = writable<Multiplier[]>([
  { id: 0, multiplier: "1x", quality: 80 }
]);
```

### **formats**

**Type**: `Writable<Format[]>`  
**Purpose**: Stores image format settings

```typescript
export let formats = writable<Format[]>([
  { id: 0, format: "webp", enabled: true }
]);
```

### **attributes**

**Type**: `Writable<Attribute[]>`  
**Purpose**: Stores HTML attributes for images

```typescript
export let attributes = writable<Attribute[]>([
  { id: 0, name: "class", value: "responsive-image" }
]);
```

### **isScreensDirty**

**Type**: `Writable<boolean>`  
**Purpose**: Tracks whether screen configuration has been modified

```typescript
export let isScreensDirty = writable<boolean>(false);
```

---

## Data Types

### **DrupalSettings**

```typescript
interface DrupalSettings {
  source: string;
  media_source: string;
  config: {
    multipliers: Multiplier[];
    screens: Screen[];
    formats: Format[];
    attributes: Attribute[];
    transform: string;
    fallback_transform: string;
  };
  aspect_ratios: string[];
  view_modes: Record<string, any>;
  app_config: {
    source: Record<string, any>;
    media_source: Record<string, any>;
    aspect_ratios: Record<string, any>;
    formats: Record<string, any>;
  };
}
```

### **AspectRatio**

```typescript
interface AspectRatio {
  id: number;
  value: string;
}
```

### **ViewMode**

```typescript
interface ViewMode {
  id: number;
  name: string;
  sizes: ViewModeSize[];
}

interface ViewModeSize {
  screen: string;
  width: string;
  aspect_ratio: string;
  enabled: boolean;
}
```

### **Screen**

```typescript
interface Screen {
  id: number;
  media_query: string;
  width: string;
}
```

### **Multiplier**

```typescript
interface Multiplier {
  id: number;
  multiplier: string;
  quality: number;
}
```

### **Format**

```typescript
interface Format {
  id: number;
  format: string;
  enabled: boolean;
}
```

### **Attribute**

```typescript
interface Attribute {
  id: number;
  name: string;
  value: string;
}
```

---

## Transformation Functions

### **transformAspectRatiosToStore**

**Purpose**: Transforms API aspect ratio data to store format

```typescript
export const transformAspectRatiosToStore = function (aspect_ratio_config: string[]) {
  aspect_ratios.set(aspect_ratio_config.map((value: string, ndx: number) => ({
    id: ndx,
    value: value,
  })));
};
```

**Parameters**:
- `aspect_ratio_config`: `string[]` - Array of aspect ratio strings from API

**Returns**: `void`

### **transformConfigToStore**

**Purpose**: Transforms API configuration data to store format

```typescript
export const transformConfigToStore = function (config: any) {
  multipliers.set(config.multipliers.map((item: any, ndx: number) => ({
    id: ndx,
    multiplier: item.multiplier,
    quality: item.quality,
  })));

  screens.set(config.screens.map((item: any, ndx: number) => ({
    id: ndx,
    media_query: item.media_query,
    width: item.width,
  })));

  formats.set(config.formats.map((item: any, ndx: number) => ({
    id: ndx,
    format: item.format,
    enabled: item.enabled,
  })));

  attributes.set(config.attributes.map((item: any, ndx: number) => ({
    id: ndx,
    name: item.name,
    value: item.value,
  })));
};
```

**Parameters**:
- `config`: `any` - Configuration object from API

**Returns**: `void`

### **transformViewModesToStore**

**Purpose**: Transforms API view mode data to store format

```typescript
export const transformViewModesToStore = function (screenValues: Screen[], viewModeConfig: Record<string, any>) {
  const transformedViewModes: ViewMode[] = [];
  
  Object.keys(viewModeConfig).forEach((viewModeName: string, ndx: number) => {
    const viewMode = viewModeConfig[viewModeName];
    const sizes: ViewModeSize[] = [];
    
    screenValues.forEach((screen: Screen) => {
      const screenName = screen.media_query;
      const size = viewMode.sizes[screenName] || {};
      
      sizes.push({
        screen: screenName,
        width: size.width || "",
        aspect_ratio: size.aspect_ratio || "",
        enabled: size.enabled !== false,
      });
    });
    
    transformedViewModes.push({
      id: ndx,
      name: viewModeName,
      sizes: sizes,
    });
  });
  
  viewModes.set(transformedViewModes);
};
```

**Parameters**:
- `screenValues`: `Screen[]` - Array of screen definitions
- `viewModeConfig`: `Record<string, any>` - View mode configuration from API

**Returns**: `void`

---

## Validation Functions

### **isValidAspectRatio**

**Purpose**: Validates aspect ratio against available options

```typescript
function isValidAspectRatio(ratio: string): boolean {
  if (!ratio || ratio.trim() === "") return false;
  const aspectRatiosConfig = $drupalSettings?.app_config?.aspect_ratios;
  if (!aspectRatiosConfig || typeof aspectRatiosConfig !== 'object') return false;
  return Object.keys(aspectRatiosConfig).includes(ratio);
}
```

**Parameters**:
- `ratio`: `string` - Aspect ratio to validate

**Returns**: `boolean` - True if valid, false otherwise

### **validateForm**

**Purpose**: Validates form data for submission

```typescript
function validateForm(): boolean {
  // Implementation depends on specific form requirements
  return true;
}
```

**Parameters**: None

**Returns**: `boolean` - True if form is valid, false otherwise

---

## Component APIs

### **FormActions Component**

**Props**: None (uses global stores)

**Methods**:
```typescript
const handleSaveValues = async function (): Promise<void>
const loadFreshConfigFromDrupal = async function (): Promise<void>
const openDialog = function (): void
```

**Events**:
- `save`: Triggered when save button is clicked
- `reset`: Triggered when reset button is clicked
- `help`: Triggered when help button is clicked

### **ViewMode Component**

**Props**:
```typescript
interface ViewModeProps {
  viewModes: Writable<any[]>;
  viewMode?: any;
}
```

**Methods**:
```typescript
function isValidAspectRatio(ratio: string): boolean
function updateViewModesStore(): void
function removeItem(id: number): void
```

**Events**:
- `change`: Triggered when view mode data changes
- `remove`: Triggered when view mode is removed

### **AspectRatioList Component**

**Props**:
```typescript
let { aspect_ratios } = $props();
```

**Methods**:
```typescript
function addItem(): void
function removeItem(id: number): void
```

**Events**:
- `add`: Triggered when aspect ratio is added
- `remove`: Triggered when aspect ratio is removed

### **ScreensList Component**

**Props**:
```typescript
let { screens } = $props();
```

**Methods**:
```typescript
function addItem(): void
function removeItem(id: number): void
```

**Events**:
- `add`: Triggered when screen is added
- `remove`: Triggered when screen is removed

### **MultiplierList Component**

**Props**:
```typescript
let { multipliers } = $props();
```

**Methods**:
```typescript
function addItem(): void
function removeItem(id: number): void
```

**Events**:
- `add`: Triggered when multiplier is added
- `remove`: Triggered when multiplier is removed

### **FormatsList Component**

**Props**:
```typescript
let { formats, drupalSettings } = $props();
```

**Methods**:
```typescript
function updateFormats(): void
```

**Events**:
- `change`: Triggered when format selection changes

### **AttributesList Component**

**Props**:
```typescript
let { attributes } = $props();
```

**Methods**:
```typescript
function addItem(): void
function removeItem(id: number): void
```

**Events**:
- `add`: Triggered when attribute is added
- `remove`: Triggered when attribute is removed

### **MainConfig Component**

**Props**:
```typescript
let { drupalSettings = $bindable() } = $props();
```

**Methods**:
```typescript
function updateSource(): void
function updateMediaSource(): void
```

**Events**:
- `change`: Triggered when configuration changes

### **MiscConfig Component**

**Props**:
```typescript
let { drupalSettings = $bindable() } = $props();
```

**Methods**:
```typescript
function updateTransform(): void
function updateFallbackTransform(): void
```

**Events**:
- `change`: Triggered when configuration changes

### **Select Component**

**Props**:
```typescript
interface Props {
  options: Record<string, any>;
  selected?: any;
  classes?: string;
}
```

**Methods**:
```typescript
function mapOptionsToSelectOptions(options: Record<string, any>): any[]
```

**Events**:
- `change`: Triggered when selection changes

### **AttributeItem Component**

**Props**:
```typescript
let { attribute, onRemove } = $props();
```

**Methods**:
```typescript
function handleRemove(): void
```

**Events**:
- `remove`: Triggered when attribute is removed

---

## Build System

### **Vite Configuration**

```typescript
// vite.config.ts
export default defineConfig({
  plugins: [svelte()],
  build: {
    outDir: '../../assets/config',
    rollupOptions: {
      output: {
        manualChunks: {
          vendor: ['svelte']
        }
      }
    }
  },
  define: {
    __BUILD_INFO__: JSON.stringify(buildInfo)
  }
});
```

### **Build Info**

```typescript
// build-info.ts
export const buildInfo = {
  version: import.meta.env.VITE_APP_VERSION || "1.0.0",
  buildId: import.meta.env.VITE_BUILD_ID || generateBuildId(),
  buildTime: import.meta.env.VITE_BUILD_TIME || new Date().toISOString(),
  commitHash: import.meta.env.VITE_COMMIT_HASH || "unknown"
};

function generateBuildId(): string {
  return `build-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
}
```

### **Environment Variables**

```bash
# Build-time variables
VITE_APP_VERSION=1.0.0
VITE_BUILD_ID=build-1234567890-abc123def
VITE_BUILD_TIME=2024-01-01T00:00:00.000Z
VITE_COMMIT_HASH=abc123def456

# Runtime variables
NODE_ENV=development
```

---

## Testing APIs

### **Test Utilities**

```typescript
// test/setup.ts
import { vi } from 'vitest';
import { cleanup } from '@testing-library/svelte';

// Mock fetch
global.fetch = vi.fn();

// Cleanup after each test
afterEach(() => {
  cleanup();
  vi.clearAllMocks();
});
```

### **Store Testing**

```typescript
// test/store.test.ts
import { get } from 'svelte/store';
import { drupalSettings, transformAspectRatiosToStore } from '../store';

describe('Store Tests', () => {
  beforeEach(() => {
    drupalSettings.set(initialValue);
  });

  it('should transform aspect ratios correctly', () => {
    const testData = ['16:9', '4:3'];
    transformAspectRatiosToStore(testData);
    
    const result = get(aspect_ratios);
    expect(result).toEqual([
      { id: 0, value: '16:9' },
      { id: 1, value: '4:3' }
    ]);
  });
});
```

### **Component Testing**

```typescript
// test/component.test.ts
import { render, fireEvent } from '@testing-library/svelte';
import Component from '../Component.svelte';

describe('Component', () => {
  it('should handle user input', async () => {
    const { getByRole } = render(Component);
    const button = getByRole('button');
    
    await fireEvent.click(button);
    // Assert expected behavior
  });
});
```

---

## Error Handling

### **API Error Handling**

```typescript
async function handleApiCall(url: string, options: RequestInit = {}) {
  try {
    const response = await fetch(url, {
      headers: {
        'Content-Type': 'application/json',
        ...options.headers
      },
      ...options
    });
    
    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }
    
    return await response.json();
  } catch (error) {
    console.error('API call failed:', error);
    throw error;
  }
}
```

### **Store Error Handling**

```typescript
// Subscribe with error handling
store.subscribe((value) => {
  try {
    // Handle store update
  } catch (error) {
    console.error('Store subscription error:', error);
  }
});
```

### **Component Error Boundaries**

```typescript
// Error boundary pattern
<script>
  let error: Error | null = null;
  
  function handleError(err: Error) {
    error = err;
    console.error('Component error:', err);
  }
</script>

{#if error}
  <div class="error">
    <p>Something went wrong: {error.message}</p>
  </div>
{:else}
  <!-- Component content -->
{/if}
```

---

## Performance APIs

### **Store Optimization**

```typescript
// Debounced store updates
import { debounce } from 'lodash';

const debouncedUpdate = debounce((value) => {
  store.set(value);
}, 300);

// Use in components
input.addEventListener('input', (e) => {
  debouncedUpdate(e.target.value);
});
```

### **Component Optimization**

```typescript
// Memoized derived values
let expensiveValue = $derived(() => {
  // Expensive calculation
  return computeExpensiveValue($store);
});

// Conditional rendering
{#if $expensiveValue}
  <ExpensiveComponent />
{/if}
```

### **Bundle Optimization**

```typescript
// Dynamic imports for code splitting
const LazyComponent = await import('./LazyComponent.svelte');

// Tree shaking friendly imports
import { specificFunction } from 'large-library';
```

---

## Migration APIs

### **Svelte 4 to 5 Migration**

```typescript
// Old Svelte 4 syntax
export let prop: string;

// New Svelte 5 syntax
let { prop = $bindable() } = $props<{
  prop?: string;
}>();

// Legacy compatibility
import { run } from 'svelte/legacy';
run(() => {
  // Legacy code here
});
```

### **Store Migration**

```typescript
// Old store usage
$store = newValue;

// New store usage
store.set(newValue);
```

---

## 📚 Additional Resources

- [Svelte 5 API Documentation](https://svelte.dev/docs)
- [TypeScript Handbook](https://www.typescriptlang.org/docs/)
- [Vitest Testing API](https://vitest.dev/api/)
- [Testing Library API](https://testing-library.com/docs/) 