# RIFT UI Config - Component Documentation

This document provides detailed information about each component in the RIFT UI Config application.

## 📋 Table of Contents

- [App.svelte](#appsvelte)
- [FormActions.svelte](#formactionssvelte)
- [ViewMode.svelte](#viewmodesvelte)
- [ViewModesList.svelte](#viewmodeslistsvelte)
- [AspectRatioList.svelte](#aspectratiolistsvelte)
- [ScreensList.svelte](#screenslistsvelte)
- [MultiplierList.svelte](#multiplierlistsvelte)
- [FormatsList.svelte](#formatslistsvelte)
- [AttributesList.svelte](#attributeslistsvelte)
- [MainConfig.svelte](#mainconfigsvelte)
- [MiscConfig.svelte](#miscconfigsvelte)
- [Select.svelte](#selectsvelte)
- [AttributeItem.svelte](#attributeitemsvelte)

---

## App.svelte

**Location**: `src/App.svelte`  
**Purpose**: Main application component that orchestrates all configuration sections

### **Props**
None - uses global stores

### **Features**
- **Collapsible Sections**: RIFT Settings and View Modes
- **Dynamic Content**: Shows warning when screens are dirty
- **Toast Notifications**: Integrated with svelte-hot-french-toast
- **Build Info**: Displays version and build information

### **Key Logic**
```typescript
// Conditional rendering based on screen state
{#if $isScreensDirty}
  <div class="alert alert-warning">Configuration changes detected...</div>
{:else}
  <ViewModesList />
{/if}
```

### **Usage**
```svelte
<App />
```

---

## FormActions.svelte

**Location**: `src/lib/partials/FormActions.svelte`  
**Purpose**: Handles form submission, data loading, and user actions

### **Props**
None - uses global stores

### **Features**
- **Save Functionality**: Submits data to `/api/rift/update`
- **Reset Functionality**: Reloads data from `/api/rift`
- **Help Dialog**: Opens documentation modal
- **Build Info Display**: Shows version and build ID
- **Loading States**: Disables buttons during operations

### **Key Methods**
```typescript
const handleSaveValues = async function () {
  // Validates form and submits data
};

const loadFreshConfigFromDrupal = async function () {
  // Loads fresh data from API
};

const openDialog = function () {
  // Opens help dialog
};
```

### **Store Subscriptions**
- `multipliers.subscribe()`: Updates drupalSettings.config.multipliers
- `screens.subscribe()`: Updates drupalSettings.config.screens
- `formats.subscribe()`: Updates drupalSettings.config.formats
- `aspect_ratios.subscribe()`: Updates drupalSettings.aspect_ratios
- `attributes.subscribe()`: Updates drupalSettings.config.attributes
- `viewModes.subscribe()`: Updates drupalSettings.view_modes

### **Usage**
```svelte
<FormActions />
```

---

## ViewMode.svelte

**Location**: `src/lib/partials/ViewMode.svelte`  
**Purpose**: Individual view mode editor with form validation

### **Props**
```typescript
interface ViewModeProps {
  viewModes: Writable<any[]>;
  viewMode?: any;
}
```

### **Features**
- **Form Validation**: Real-time validation of aspect ratios
- **Dynamic Fields**: Screen-specific width and aspect ratio inputs
- **Error Handling**: Visual feedback for invalid inputs
- **Two-way Binding**: Updates parent store automatically
- **Collapsible Interface**: Expandable view mode sections

### **Key Methods**
```typescript
function isValidAspectRatio(ratio: string): boolean {
  // Validates aspect ratio against drupalSettings.app_config.aspect_ratios
}

function updateViewModesStore() {
  // Triggers store update to save changes
}

function removeItem(id: number) {
  // Removes view mode from list
}
```

### **Validation Logic**
```typescript
let localFormValid = $derived($isScreensDirty ? true : viewMode.sizes.every((size: any) => {
  if (!size.enabled) return true;
  return size.width && size.width.trim() !== "" && 
         size.aspect_ratio && size.aspect_ratio.trim() !== "" && 
         isValidAspectRatio(size.aspect_ratio);
}));
```

### **Usage**
```svelte
<ViewMode bind:viewMode={item} viewModes={viewModes} />
```

---

## ViewModesList.svelte

**Location**: `src/lib/partials/ViewModesList.svelte`  
**Purpose**: Container for managing multiple view modes

### **Props**
None - uses global stores

### **Features**
- **Dynamic List**: Renders ViewMode components for each view mode
- **Add Functionality**: Button to create new view modes
- **Store Integration**: Uses global viewModes store

### **Key Methods**
```typescript
function addItem() {
  // Creates new view mode with default values
}
```

### **Usage**
```svelte
<ViewModesList />
```

---

## AspectRatioList.svelte

**Location**: `src/lib/partials/AspectRatioList.svelte`  
**Purpose**: Manages aspect ratio definitions

### **Props**
```typescript
let { aspect_ratios } = $props();
```

### **Features**
- **Add/Remove**: Dynamic aspect ratio management
- **Form Validation**: Ensures valid aspect ratio format
- **Store Integration**: Updates aspect_ratios store
- **Dirty State**: Marks form as dirty when changes are made

### **Key Methods**
```typescript
function addItem() {
  // Adds new aspect ratio to list
}

function removeItem(id: number) {
  // Removes aspect ratio by ID
}
```

### **Usage**
```svelte
<AspectRatioList aspect_ratios={aspect_ratios} />
```

---

## ScreensList.svelte

**Location**: `src/lib/partials/ScreensList.svelte`  
**Purpose**: Manages screen breakpoint definitions

### **Props**
```typescript
let { screens } = $props();
```

### **Features**
- **Screen Management**: Add/remove screen breakpoints
- **Media Queries**: Configure responsive breakpoints
- **Width Settings**: Set screen widths for responsive design
- **Store Integration**: Updates screens store

### **Key Methods**
```typescript
function addItem() {
  // Adds new screen breakpoint
}

function removeItem(id: number) {
  // Removes screen breakpoint by ID
}
```

### **Usage**
```svelte
<ScreensList screens={screens} />
```

---

## MultiplierList.svelte

**Location**: `src/lib/partials/MultiplierList.svelte`  
**Purpose**: Manages image multiplier and quality settings

### **Props**
```typescript
let { multipliers } = $props();
```

### **Features**
- **Multiplier Management**: Add/remove image multipliers
- **Quality Settings**: Configure quality for each multiplier
- **Store Integration**: Updates multipliers store
- **Form Validation**: Ensures valid multiplier format

### **Key Methods**
```typescript
function addItem() {
  // Adds new multiplier with default quality
}

function removeItem(id: number) {
  // Removes multiplier by ID
}
```

### **Usage**
```svelte
<MultiplierList multipliers={multipliers} />
```

---

## FormatsList.svelte

**Location**: `src/lib/partials/FormatsList.svelte`  
**Purpose**: Manages image format settings

### **Props**
```typescript
let { formats, drupalSettings } = $props();
```

### **Features**
- **Format Selection**: Choose from available image formats
- **API Integration**: Uses drupalSettings.app_config.formats
- **Store Integration**: Updates formats store
- **Dynamic Options**: Loads formats from API

### **Usage**
```svelte
<FormatsList {formats} drupalSettings={$drupalSettings} />
```

---

## AttributesList.svelte

**Location**: `src/lib/partials/AttributesList.svelte`  
**Purpose**: Manages HTML attributes for images

### **Props**
```typescript
let { attributes } = $props();
```

### **Features**
- **Attribute Management**: Add/remove HTML attributes
- **Key-Value Pairs**: Name and value for each attribute
- **Store Integration**: Updates attributes store
- **Form Validation**: Ensures valid attribute names

### **Key Methods**
```typescript
function addItem() {
  // Adds new attribute
}

function removeItem(id: number) {
  // Removes attribute by ID
}
```

### **Usage**
```svelte
<AttributesList bind:attributes={$attributes} />
```

---

## MainConfig.svelte

**Location**: `src/lib/partials/MainConfig.svelte`  
**Purpose**: Primary configuration settings

### **Props**
```typescript
let { drupalSettings = $bindable() } = $props();
```

### **Features**
- **Source Selection**: Choose image source
- **Media Source**: Configure media source settings
- **API Integration**: Uses drupalSettings.app_config
- **Two-way Binding**: Updates drupalSettings store

### **Usage**
```svelte
<MainConfig bind:drupalSettings={$drupalSettings} />
```

---

## MiscConfig.svelte

**Location**: `src/lib/partials/MiscConfig.svelte`  
**Purpose**: Miscellaneous configuration settings

### **Props**
```typescript
let { drupalSettings = $bindable() } = $props();
```

### **Features**
- **Transform Settings**: Configure image transforms
- **Fallback Settings**: Set fallback transform
- **Two-way Binding**: Updates drupalSettings store
- **Form Validation**: Ensures valid transform syntax

### **Usage**
```svelte
<MiscConfig bind:drupalSettings={$drupalSettings} />
```

---

## Select.svelte

**Location**: `src/lib/partials/Select.svelte`  
**Purpose**: Reusable dropdown component

### **Props**
```typescript
interface Props {
  options: Record<string, any>;
  selected?: any;
  classes?: string;
}
```

### **Features**
- **Dynamic Options**: Renders options from object
- **Custom Styling**: Configurable CSS classes
- **Two-way Binding**: Updates selected value
- **Accessibility**: Proper ARIA attributes

### **Key Methods**
```typescript
function mapOptionsToSelectOptions(options: Record<string, any>): any[] {
  // Converts object to array for rendering
}
```

### **Usage**
```svelte
<Select 
  options={drupalSettings.app_config.source} 
  bind:selected={drupalSettings.source} 
/>
```

---

## AttributeItem.svelte

**Location**: `src/lib/partials/AttributeItem.svelte`  
**Purpose**: Individual attribute editor component

### **Props**
```typescript
let { attribute, onRemove } = $props();
```

### **Features**
- **Name/Value Inputs**: Edit attribute name and value
- **Remove Functionality**: Delete attribute from list
- **Form Validation**: Ensures valid attribute format
- **Event Handling**: Notifies parent of changes

### **Key Methods**
```typescript
function handleRemove() {
  // Triggers onRemove callback
}
```

### **Usage**
```svelte
<AttributeItem 
  bind:attribute={item} 
  onRemove={() => removeItem(item.id)} 
/>
```

---

## 🔧 Component Development Guidelines

### **Creating New Components**

1. **File Naming**: Use PascalCase (e.g., `NewComponent.svelte`)
2. **Location**: Place in `src/lib/partials/` for reusable components
3. **Props**: Use Svelte 5 runes with proper TypeScript typing
4. **Stores**: Subscribe to global stores when needed
5. **Testing**: Write tests for new components

### **Component Patterns**

#### **Form Components**
```typescript
// Use $bindable() for two-way binding
let { value = $bindable() } = $props();

// Use $derived() for computed values
let isValid = $derived(validationLogic);
```

#### **List Components**
```typescript
// Use $each for rendering lists
{#each items as item (item.id)}
  <ItemComponent bind:item={item} />
{/each}
```

#### **Store Integration**
```typescript
// Subscribe to store changes
store.subscribe((value) => {
  // Handle store updates
});
```

### **Best Practices**

1. **TypeScript**: Use strict typing for all props and methods
2. **Accessibility**: Include proper ARIA attributes
3. **Error Handling**: Provide meaningful error messages
4. **Performance**: Use reactive statements efficiently
5. **Testing**: Write comprehensive tests for all functionality

---

## 📚 Additional Resources

- [Svelte 5 Component Documentation](https://svelte.dev/docs)
- [TypeScript with Svelte](https://svelte.dev/docs/typescript)
- [Testing Svelte Components](https://testing-library.com/docs/svelte-testing-library/intro/)
- [DaisyUI Components](https://daisyui.com/components/) 