# RIFT UI Config - Development Guide

This guide provides comprehensive information for developers working on the RIFT UI Config application.

## 🚀 Quick Start

### **Prerequisites**
```bash
# Install Bun (recommended)
curl -fsSL https://bun.sh/install | bash

# Or use Node.js 18+
node --version
```

### **Setup**
```bash
# Clone and navigate to project
cd modules/rift_ui/app/config

# Install dependencies
bun install

# Start development server
bun run dev
```

### **Available Scripts**
```bash
bun run dev          # Development server
bun run build        # Production build
bun run preview      # Preview production build
bun test             # Run tests
bun test --run       # Run tests once
bun test --ui        # Run tests with UI
bun run check        # Type checking
```

---

## 🧠 Core Concepts

### **Svelte 5 Runes**

The application uses Svelte 5's new runes system:

```typescript
// Props with runes
let { value = $bindable() } = $props<{
  value?: string;
}>();

// Derived values
let computed = $derived(expression);

// State
let count = $state(0);

// Legacy compatibility
import { run } from 'svelte/legacy';
```

### **Store Management**

Centralized state management with Svelte stores:

```typescript
// Store definition
export let drupalSettings = writable<DrupalSettings>({...});

// Store subscription
drupalSettings.subscribe((value) => {
  // Handle store updates
});

// Store transformation
export const transformConfigToStore = function (config: any) {
  // Transform API data to store format
};
```

### **Data Flow**

1. **API Load**: Data loaded from `/api/rift`
2. **Transform**: Raw data transformed to store format
3. **Component Update**: Components react to store changes
4. **User Input**: Changes update stores
5. **API Save**: Stores transformed back to API format

---

## 🔧 Development Patterns

### **Component Structure**

```typescript
<script lang="ts">
  // Imports
  import { run } from 'svelte/legacy';
  import { someStore } from '../store';
  
  // Props with runes
  let { prop = $bindable() } = $props<{
    prop?: any;
  }>();
  
  // Derived values
  let isValid = $derived(validationLogic);
  
  // Methods
  function handleAction() {
    // Action logic
  }
  
  // Legacy compatibility
  run(() => {
    // Legacy code
  });
</script>

<!-- Template -->
<div class="component">
  <!-- Content -->
</div>
```

### **Form Handling**

```typescript
// Two-way binding
<input bind:value={formData.field} />

// Validation
let isValid = $derived(
  formData.field && formData.field.trim() !== ""
);

// Error display
<input class={!isValid ? 'input-error' : ''} />
```

### **Store Integration**

```typescript
// Subscribe to store changes
store.subscribe((value) => {
  // Handle updates
});

// Update store
store.set(newValue);

// Transform data
export const transformData = function (apiData: any) {
  store.set(transform(apiData));
};
```

---

## 🧪 Testing

### **Test Structure**

```typescript
import { describe, it, expect, beforeEach } from "vitest";
import { get } from "svelte/store";
import { someStore, transformFunction } from "../store";

describe("Feature Tests", () => {
  beforeEach(() => {
    // Reset stores
    someStore.set(initialValue);
  });

  it("should transform data correctly", () => {
    // Test implementation
    const result = transformFunction(testData);
    expect(result).toEqual(expectedValue);
  });
});
```

### **Component Testing**

```typescript
import { render, fireEvent } from '@testing-library/svelte';
import Component from './Component.svelte';

describe("Component", () => {
  it("should handle user input", async () => {
    const { getByRole } = render(Component);
    const button = getByRole('button');
    
    await fireEvent.click(button);
    // Assert expected behavior
  });
});
```

### **Running Tests**

```bash
# Run all tests
bun test

# Run specific test
bun test store.test.ts

# Run with coverage
bun test --coverage

# Run with UI
bun test --ui
```

---

## 🐛 Troubleshooting

### **Common Issues**

#### **"e.subscribe is not a function" Error**
**Problem**: Passing store value instead of store object
```typescript
// ❌ Wrong
<Component store={$store} />

// ✅ Correct
<Component store={store} />
```

**Solution**: Always pass the store object, not its value

#### **Form Jumping to Top**
**Problem**: Default form submission behavior
```typescript
// ❌ Wrong
<button onclick={handleClick}>Save</button>

// ✅ Correct
<button type="button" onclick={handleClick}>Save</button>
```

**Solution**: Add `type="button"` to prevent form submission

#### **TypeScript Errors with Runes**
**Problem**: Svelte 5 runes compatibility
```typescript
// ❌ Wrong
export let prop: string;

// ✅ Correct
let { prop = $bindable() } = $props<{
  prop?: string;
}>();
```

**Solution**: Use `$props()` instead of `export let`

#### **Store Not Updating**
**Problem**: Incorrect store usage
```typescript
// ❌ Wrong
$store = newValue;

// ✅ Correct
store.set(newValue);
```

**Solution**: Use `.set()` method to update stores

### **Debugging Tips**

1. **Check Browser Console**: Look for runtime errors
2. **Use DevTools**: Inspect component state and props
3. **Add Console Logs**: Temporary debugging
4. **Check Network Tab**: Monitor API calls
5. **Run Tests**: Verify functionality

### **Performance Issues**

1. **Store Subscriptions**: Ensure proper cleanup
2. **Reactive Statements**: Use `$derived()` efficiently
3. **Component Re-renders**: Minimize unnecessary updates
4. **Bundle Size**: Monitor build output size

---

## 🔄 Data Transformation

### **API to Store**

```typescript
// Transform API response to store format
export const transformAspectRatiosToStore = function (aspect_ratio_config: string[]) {
  aspect_ratios.set(aspect_ratio_config.map((value: string, ndx: number) => ({
    id: ndx,
    value: value,
  })));
};
```

### **Store to API**

```typescript
// Transform store data back to API format
aspect_ratios.subscribe((value: any[]) => {
  const currentSettings = $drupalSettings;
  currentSettings["aspect_ratios"] = value.map((v: any) => v.value);
  drupalSettings.set(currentSettings);
});
```

### **Validation Functions**

```typescript
// Validate aspect ratios
function isValidAspectRatio(ratio: string): boolean {
  if (!ratio || ratio.trim() === "") return false;
  const aspectRatiosConfig = $drupalSettings?.app_config?.aspect_ratios;
  if (!aspectRatiosConfig || typeof aspectRatiosConfig !== 'object') return false;
  return Object.keys(aspectRatiosConfig).includes(ratio);
}
```

---

## 🎨 Styling Guidelines

### **Tailwind CSS**

```html
<!-- Responsive design -->
<div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3">

<!-- Form styling -->
<input class="input input-bordered input-sm w-full" />

<!-- Conditional styling -->
<div class="alert {isError ? 'alert-error' : 'alert-success'}">
```

### **DaisyUI Components**

```html
<!-- Buttons -->
<button class="btn btn-primary">Primary</button>
<button class="btn btn-secondary">Secondary</button>

<!-- Forms -->
<div class="form-control">
  <label class="label">Label</label>
  <input class="input input-bordered" />
</div>

<!-- Alerts -->
<div class="alert alert-info">
  <span>Information message</span>
</div>
```

### **Custom CSS**

```css
/* Component-specific styles */
.component {
  @apply p-4 border rounded-lg;
}

/* Responsive utilities */
@media (max-width: 768px) {
  .mobile-hidden {
    @apply hidden;
  }
}
```

---

## 🔧 Configuration

### **Environment Variables**

```bash
# Development
VITE_APP_VERSION=1.0.0
VITE_BUILD_ID=dev-123
VITE_BUILD_TIME=2024-01-01T00:00:00Z
VITE_COMMIT_HASH=abc123

# Production
NODE_ENV=production
VITE_API_BASE_URL=https://api.example.com
```

### **Build Configuration**

```typescript
// vite.config.ts
export default defineConfig({
  plugins: [svelte()],
  build: {
    outDir: '../../assets/config',
    rollupOptions: {
      output: {
        manualChunks: {
          vendor: ['svelte']
        }
      }
    }
  }
});
```

### **TypeScript Configuration**

```json
{
  "compilerOptions": {
    "strict": true,
    "allowJs": true,
    "checkJs": false,
    "esModuleInterop": true,
    "skipLibCheck": true,
    "forceConsistentCasingInFileNames": true
  }
}
```

---

## 🚀 Deployment

### **Build Process**

```bash
# Production build
bun run build

# Output location
../../assets/config/
```

### **Integration with Drupal**

1. **Build Application**: `bun run build`
2. **Copy Assets**: Copy to Drupal theme directory
3. **Include in Template**: Add to Drupal template
4. **Configure API**: Set up API endpoints

### **Environment Setup**

```bash
# Development
NODE_ENV=development bun run dev

# Production
NODE_ENV=production bun run build
```

---

## 📚 Best Practices

### **Code Organization**

1. **File Structure**: Follow established patterns
2. **Component Naming**: Use PascalCase
3. **Function Naming**: Use camelCase
4. **Type Definitions**: Use TypeScript interfaces

### **State Management**

1. **Centralized Stores**: Use global stores for shared state
2. **Local State**: Use component state for local data
3. **Store Subscriptions**: Clean up subscriptions properly
4. **Data Transformation**: Separate API and store formats

### **Error Handling**

1. **Try-Catch Blocks**: Handle async operations
2. **User Feedback**: Show meaningful error messages
3. **Validation**: Validate user input
4. **Fallbacks**: Provide default values

### **Performance**

1. **Lazy Loading**: Load components on demand
2. **Memoization**: Use `$derived()` for expensive calculations
3. **Bundle Splitting**: Split code into chunks
4. **Tree Shaking**: Remove unused code

### **Accessibility**

1. **ARIA Attributes**: Include proper accessibility attributes
2. **Keyboard Navigation**: Support keyboard-only navigation
3. **Screen Readers**: Ensure screen reader compatibility
4. **Color Contrast**: Maintain proper color contrast

---

## 🤝 Contributing

### **Development Workflow**

1. **Create Branch**: `git checkout -b feature/new-feature`
2. **Make Changes**: Follow coding standards
3. **Write Tests**: Add tests for new functionality
4. **Run Tests**: Ensure all tests pass
5. **Submit PR**: Create pull request with description

### **Code Review Checklist**

- [ ] Code follows TypeScript standards
- [ ] Tests are included and passing
- [ ] No console.log statements in production
- [ ] Proper error handling
- [ ] Accessibility considerations
- [ ] Performance impact assessed

### **Documentation**

- [ ] Update README.md for new features
- [ ] Add component documentation
- [ ] Update API documentation
- [ ] Include usage examples

---

## 📖 Additional Resources

- [Svelte 5 Documentation](https://svelte.dev/docs)
- [TypeScript Handbook](https://www.typescriptlang.org/docs/)
- [Tailwind CSS Documentation](https://tailwindcss.com/docs)
- [DaisyUI Components](https://daisyui.com/components/)
- [Vitest Testing Guide](https://vitest.dev/guide/)
- [Testing Library](https://testing-library.com/docs/)

---

## 🆘 Getting Help

### **Internal Resources**
- Check existing documentation
- Review test files for examples
- Look at similar components

### **External Resources**
- Svelte Discord community
- Stack Overflow with `svelte` tag
- GitHub issues for dependencies

### **Team Support**
- Create issue with detailed description
- Include error messages and steps to reproduce
- Provide browser and environment information 