# RIFT UI Config - Quick Reference

A quick reference guide for developers working with the RIFT UI Config application.

## 🚀 Quick Commands

### **Development**
```bash
# Start development server
bun run dev

# Build for production
bun run build

# Preview production build
bun run preview

# Type checking
bun run check
```

### **Testing**
```bash
# Run all tests
bun test

# Run tests once
bun test --run

# Run tests with UI
bun test --ui

# Run tests with coverage
bun test --coverage

# Run specific test file
bun test store.test.ts
```

### **Package Management**
```bash
# Install dependencies
bun install

# Add new dependency
bun add package-name

# Add dev dependency
bun add -d package-name

# Remove dependency
bun remove package-name
```

---

## 📝 Common Patterns

### **Svelte 5 Runes**
```typescript
// Props
let { value = $bindable() } = $props<{
  value?: string;
}>();

// Derived values
let computed = $derived(expression);

// State
let count = $state(0);

// Legacy compatibility
import { run } from 'svelte/legacy';
run(() => {
  // Legacy code
});
```

### **Store Usage**
```typescript
// Create store
export let store = writable<Type>(initialValue);

// Subscribe to store
store.subscribe((value) => {
  // Handle updates
});

// Update store
store.set(newValue);

// Get store value
const value = get(store);
```

### **Component Structure**
```typescript
<script lang="ts">
  // Imports
  import { run } from 'svelte/legacy';
  
  // Props
  let { prop = $bindable() } = $props<{
    prop?: any;
  }>();
  
  // Derived values
  let isValid = $derived(validationLogic);
  
  // Methods
  function handleAction() {
    // Action logic
  }
  
  // Legacy compatibility
  run(() => {
    // Legacy code
  });
</script>

<!-- Template -->
<div class="component">
  <!-- Content -->
</div>
```

### **Form Handling**
```typescript
// Two-way binding
<input bind:value={formData.field} />

// Validation
let isValid = $derived(
  formData.field && formData.field.trim() !== ""
);

// Error display
<input class={!isValid ? 'input-error' : ''} />

// Prevent form submission
<button type="button" onclick={handleClick}>Save</button>
```

### **List Rendering**
```typescript
{#each items as item (item.id)}
  <ItemComponent bind:item={item} />
{/each}

// With index
{#each items as item, index (item.id)}
  <ItemComponent {item} {index} />
{/each}
```

### **Conditional Rendering**
```typescript
{#if condition}
  <Component />
{:else if otherCondition}
  <OtherComponent />
{:else}
  <DefaultComponent />
{/if}
```

---

## 🔧 Troubleshooting

### **Common Errors**

#### **"e.subscribe is not a function"**
**Problem**: Passing store value instead of store object
```typescript
// ❌ Wrong
<Component store={$store} />

// ✅ Correct
<Component store={store} />
```

#### **Form Jumping to Top**
**Problem**: Missing `type="button"`
```typescript
// ❌ Wrong
<button onclick={handleClick}>Save</button>

// ✅ Correct
<button type="button" onclick={handleClick}>Save</button>
```

#### **TypeScript Errors with Runes**
**Problem**: Using old Svelte 4 syntax
```typescript
// ❌ Wrong
export let prop: string;

// ✅ Correct
let { prop = $bindable() } = $props<{
  prop?: string;
}>();
```

#### **Store Not Updating**
**Problem**: Incorrect store usage
```typescript
// ❌ Wrong
$store = newValue;

// ✅ Correct
store.set(newValue);
```

### **Debugging Tips**

1. **Check Browser Console**: Look for runtime errors
2. **Use DevTools**: Inspect component state and props
3. **Add Console Logs**: Temporary debugging
4. **Check Network Tab**: Monitor API calls
5. **Run Tests**: Verify functionality

### **Performance Issues**

1. **Store Subscriptions**: Ensure proper cleanup
2. **Reactive Statements**: Use `$derived()` efficiently
3. **Component Re-renders**: Minimize unnecessary updates
4. **Bundle Size**: Monitor build output size

---

## 🎨 Styling Quick Reference

### **Tailwind CSS Classes**
```html
<!-- Layout -->
<div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3">
<div class="flex flex-col md:flex-row">
<div class="hidden md:block">

<!-- Spacing -->
<div class="p-4 m-2">
<div class="space-y-4">

<!-- Colors -->
<div class="text-red-500 bg-blue-100">
<div class="border border-gray-300">

<!-- Forms -->
<input class="input input-bordered input-sm w-full" />
<button class="btn btn-primary">Button</button>
```

### **DaisyUI Components**
```html
<!-- Buttons -->
<button class="btn btn-primary">Primary</button>
<button class="btn btn-secondary">Secondary</button>
<button class="btn btn-outline">Outline</button>

<!-- Forms -->
<div class="form-control">
  <label class="label">Label</label>
  <input class="input input-bordered" />
</div>

<!-- Alerts -->
<div class="alert alert-info">
  <span>Information message</span>
</div>

<!-- Cards -->
<div class="card bg-base-100 shadow-xl">
  <div class="card-body">
    <h2 class="card-title">Title</h2>
    <p>Content</p>
  </div>
</div>
```

---

## 📊 Data Transformation

### **API to Store**
```typescript
// Transform aspect ratios
export const transformAspectRatiosToStore = function (aspect_ratio_config: string[]) {
  aspect_ratios.set(aspect_ratio_config.map((value: string, ndx: number) => ({
    id: ndx,
    value: value,
  })));
};
```

### **Store to API**
```typescript
// Transform store data back to API format
aspect_ratios.subscribe((value: any[]) => {
  const currentSettings = $drupalSettings;
  currentSettings["aspect_ratios"] = value.map((v: any) => v.value);
  drupalSettings.set(currentSettings);
});
```

### **Validation**
```typescript
// Validate aspect ratios
function isValidAspectRatio(ratio: string): boolean {
  if (!ratio || ratio.trim() === "") return false;
  const aspectRatiosConfig = $drupalSettings?.app_config?.aspect_ratios;
  if (!aspectRatiosConfig || typeof aspectRatiosConfig !== 'object') return false;
  return Object.keys(aspectRatiosConfig).includes(ratio);
}
```

---

## 🧪 Testing Quick Reference

### **Test Structure**
```typescript
import { describe, it, expect, beforeEach } from "vitest";
import { get } from "svelte/store";
import { someStore, transformFunction } from "../store";

describe("Feature Tests", () => {
  beforeEach(() => {
    // Reset stores
    someStore.set(initialValue);
  });

  it("should transform data correctly", () => {
    const result = transformFunction(testData);
    expect(result).toEqual(expectedValue);
  });
});
```

### **Component Testing**
```typescript
import { render, fireEvent } from '@testing-library/svelte';
import Component from './Component.svelte';

describe("Component", () => {
  it("should handle user input", async () => {
    const { getByRole } = render(Component);
    const button = getByRole('button');
    
    await fireEvent.click(button);
    // Assert expected behavior
  });
});
```

### **Mocking**
```typescript
// Mock fetch
global.fetch = vi.fn();

// Mock store
vi.mock('../store', () => ({
  someStore: writable(mockValue)
}));
```

---

## 🔧 Configuration

### **Environment Variables**
```bash
# Development
VITE_APP_VERSION=1.0.0
VITE_BUILD_ID=dev-123
VITE_BUILD_TIME=2024-01-01T00:00:00Z
VITE_COMMIT_HASH=abc123

# Production
NODE_ENV=production
```

### **Build Output**
```
../../assets/config/
├── index.html
├── assets/
│   ├── index-[hash].js
│   └── index-[hash].css
```

---

## 📚 File Structure

```
src/
├── App.svelte              # Main application
├── main.ts                 # Entry point
├── app.css                 # Global styles
├── lib/
│   ├── store.ts           # State management
│   ├── build-info.ts      # Build metadata
│   └── partials/          # Components
│       ├── FormActions.svelte
│       ├── ViewMode.svelte
│       ├── ViewModesList.svelte
│       ├── AspectRatioList.svelte
│       ├── ScreensList.svelte
│       ├── MultiplierList.svelte
│       ├── FormatsList.svelte
│       ├── AttributesList.svelte
│       ├── MainConfig.svelte
│       ├── MiscConfig.svelte
│       ├── Select.svelte
│       └── AttributeItem.svelte
└── test/
    ├── store.test.ts
    ├── form-submission.test.ts
    ├── setup.ts
    └── README.md
```

---

## 🚀 Deployment

### **Build Process**
```bash
# Production build
bun run build

# Output location
../../assets/config/
```

### **Integration with Drupal**
1. Build application: `bun run build`
2. Copy assets to Drupal theme
3. Include in Drupal template
4. Configure API endpoints

---

## 📖 Additional Resources

- [Svelte 5 Documentation](https://svelte.dev/docs)
- [TypeScript Handbook](https://www.typescriptlang.org/docs/)
- [Tailwind CSS Documentation](https://tailwindcss.com/docs)
- [DaisyUI Components](https://daisyui.com/components/)
- [Vitest Testing Guide](https://vitest.dev/guide/)
- [Testing Library](https://testing-library.com/docs/)

---

## 🆘 Getting Help

### **Internal Resources**
- Check existing documentation
- Review test files for examples
- Look at similar components

### **External Resources**
- Svelte Discord community
- Stack Overflow with `svelte` tag
- GitHub issues for dependencies

### **Team Support**
- Create issue with detailed description
- Include error messages and steps to reproduce
- Provide browser and environment information 