# RIFT UI Config - Developer Documentation

A modern Svelte 5 application for configuring RIFT (Responsive Image Formatter Tools) settings in Drupal. This application provides a comprehensive UI for managing responsive image configurations, view modes, aspect ratios, and more.

## 🏗️ Architecture Overview

### **Technology Stack**
- **Frontend**: Svelte 5.36.0 with Runes mode
- **Language**: TypeScript 5.5.3
- **Build Tool**: Vite 7.0.4
- **Styling**: Tailwind CSS 3.4.9 + DaisyUI 4.12.10
- **Testing**: Vitest 1.0.0 + Testing Library
- **Package Manager**: Bun

### **Project Structure**
```
src/
├── App.svelte              # Main application component
├── main.ts                 # Application entry point
├── app.css                 # Global styles
├── lib/
│   ├── store.ts           # Central state management
│   ├── build-info.ts      # Build metadata & versioning
│   └── partials/          # Reusable UI components
│       ├── FormActions.svelte     # Save/Reset/Help buttons
│       ├── ViewMode.svelte        # Individual view mode editor
│       ├── ViewModesList.svelte   # List of view modes
│       ├── AspectRatioList.svelte # Aspect ratio management
│       ├── ScreensList.svelte     # Screen breakpoint management
│       ├── MultiplierList.svelte  # Image multiplier settings
│       ├── FormatsList.svelte     # Image format settings
│       ├── AttributesList.svelte  # HTML attributes management
│       ├── MainConfig.svelte      # Primary configuration
│       ├── MiscConfig.svelte      # Miscellaneous settings
│       ├── Select.svelte          # Reusable select component
│       └── AttributeItem.svelte   # Individual attribute editor
└── test/
    ├── store.test.ts       # Store transformation tests
    ├── form-submission.test.ts # Form integration tests
    ├── setup.ts            # Test configuration
    └── README.md           # Testing documentation
```

## 🚀 Development Workflow

### **Prerequisites**
- Node.js 18+ or Bun
- Modern browser with ES6+ support

### **Installation**
```bash
# Install dependencies
bun install

# Or with npm
npm install
```

### **Development Commands**
```bash
# Start development server
bun run dev

# Build for production
bun run build

# Preview production build
bun run preview

# Run tests
bun test

# Run tests once
bun test --run

# Run tests with UI
bun test --ui

# Run tests with coverage
bun test --coverage

# Type checking
bun run check
```

### **Development Server**
The development server runs on `http://localhost:5173/` and includes:
- Hot Module Replacement (HMR)
- TypeScript compilation
- Tailwind CSS processing
- Source maps for debugging

## 🧠 Key Concepts

### **State Management**
The application uses Svelte stores for centralized state management:

```typescript
// Core stores
export let drupalSettings = writable<DrupalSettings>({...});
export let aspect_ratios = writable<AspectRatio[]>([...]);
export let viewModes = writable<ViewMode[]>([]);
export let screens = writable<Screen[]>([]);
export let formats = writable<Format[]>([]);
export let multipliers = writable<Multiplier[]>([]);
export let attributes = writable<Attribute[]>([]);
```

### **Data Flow**
1. **API Integration**: Data is loaded from Drupal via `/api/rift` endpoint
2. **Store Transformation**: Raw API data is transformed into Svelte store format
3. **Component Updates**: Components subscribe to stores for reactive updates
4. **Form Submission**: Changes are sent back to Drupal via `/api/rift/update`

### **Svelte 5 Runes**
The application uses Svelte 5's new runes system:

```typescript
// Props with runes
let { viewModes, viewMode = $bindable() } = $props<{
    viewModes: Writable<any[]>;
    viewMode?: any;
}>();

// Derived values
let localFormValid = $derived(/* validation logic */);

// Legacy compatibility
import { run } from 'svelte/legacy';
```

## 🧪 Testing Strategy

### **Test Coverage**
- **Store Tests**: Data transformation and validation
- **Form Tests**: User interactions and submissions
- **Integration Tests**: End-to-end workflows

### **Running Tests**
```bash
# Run all tests
bun test

# Run specific test file
bun test store.test.ts

# Run tests in watch mode
bun test --watch

# Generate coverage report
bun test --coverage
```

### **Test Structure**
```typescript
describe("Store Tests", () => {
  beforeEach(() => {
    // Reset stores to initial state
  });

  it("should transform aspect ratios correctly", () => {
    // Test implementation
  });
});
```

## 🔧 Configuration

### **Build Configuration**
- **Vite**: Optimized for Svelte 5 with legacy compatibility
- **TypeScript**: Strict mode with Svelte-specific types
- **Tailwind**: Custom configuration with DaisyUI components
- **Testing**: JSDOM environment with Testing Library

### **Environment Variables**
```bash
VITE_APP_VERSION=1.0.0
VITE_BUILD_ID=auto-generated
VITE_BUILD_TIME=timestamp
VITE_COMMIT_HASH=git-hash
```

### **Build Output**
- **HTML**: 0.46 kB (gzipped: 0.30 kB)
- **CSS**: 57.07 kB (gzipped: 9.99 kB)
- **JS**: 73.13 kB (gzipped: 24.74 kB)
- **Total**: ~130 kB (gzipped: ~35 kB)

## 🎯 Component Architecture

### **Form Components**
Each configuration section is a separate component:

- **MainConfig**: Primary settings (source, media source)
- **AspectRatioList**: Manage aspect ratios
- **ScreensList**: Configure screen breakpoints
- **MultiplierList**: Set image multipliers and quality
- **FormatsList**: Choose image formats
- **AttributesList**: HTML attributes for images
- **MiscConfig**: Additional settings

### **View Mode Components**
- **ViewModesList**: Container for view modes
- **ViewMode**: Individual view mode editor
- **FormActions**: Save/Reset/Help functionality

### **Reusable Components**
- **Select**: Dropdown component with options
- **AttributeItem**: Individual attribute editor

## 🔄 Data Transformation

### **API to Store Transformation**
```typescript
// Transform API response to store format
export const transformAspectRatiosToStore = function (aspect_ratio_config: string[]) {
  aspect_ratios.set(aspect_ratio_config.map((value: string, ndx: number) => ({
    id: ndx,
    value: value,
  })));
};
```

### **Store to API Transformation**
```typescript
// Transform store data back to API format
aspect_ratios.subscribe((value: any[]) => {
  const currentSettings = $drupalSettings;
  currentSettings["aspect_ratios"] = value.map((v: any) => v.value);
  drupalSettings.set(currentSettings);
});
```

## 🐛 Troubleshooting

### **Common Issues**

#### **"e.subscribe is not a function" Error**
**Cause**: Passing store value instead of store object
**Solution**: Use `viewModes` instead of `$viewModes` for bindings

#### **Form Jumping to Top**
**Cause**: Default form submission behavior
**Solution**: Add `type="button"` to action buttons

#### **TypeScript Errors**
**Cause**: Svelte 5 runes compatibility
**Solution**: Use `$props()` instead of `export let`

### **Debugging Tips**
1. **Check Browser Console**: For runtime errors
2. **Use DevTools**: Inspect component state
3. **Run Tests**: Verify functionality
4. **Check Network Tab**: Monitor API calls

### **Performance Optimization**
- **Lazy Loading**: Components load on demand
- **Store Subscriptions**: Automatic cleanup
- **Bundle Splitting**: Code splitting for better performance

## 📚 API Reference

### **Store Methods**
```typescript
// Transform functions
transformAspectRatiosToStore(aspect_ratio_config: string[])
transformConfigToStore(config: any)
transformViewModesToStore(screenValues: Screen[], viewModeConfig: Record<string, any>)

// Store subscriptions
drupalSettings.subscribe(callback)
aspect_ratios.subscribe(callback)
viewModes.subscribe(callback)
```

### **Component Props**
```typescript
// ViewMode component
interface ViewModeProps {
  viewModes: Writable<any[]>;
  viewMode?: any;
}

// FormActions component
interface FormActionsProps {
  // No props - uses global stores
}
```

## 🚀 Deployment

### **Build Process**
```bash
# Production build
bun run build

# Output location
../../assets/config/
```

### **Integration with Drupal**
1. Build the application
2. Copy assets to Drupal theme
3. Include in Drupal template
4. Configure API endpoints

### **Environment Setup**
```bash
# Development
NODE_ENV=development bun run dev

# Production
NODE_ENV=production bun run build
```

## 🤝 Contributing

### **Development Guidelines**
1. **TypeScript**: Use strict typing
2. **Testing**: Write tests for new features
3. **Svelte 5**: Use runes where appropriate
4. **Styling**: Follow Tailwind + DaisyUI patterns

### **Code Style**
- **Indentation**: 2 spaces
- **Quotes**: Double quotes for strings
- **Semicolons**: Required
- **TypeScript**: Strict mode enabled

### **Git Workflow**
1. Create feature branch
2. Make changes
3. Run tests
4. Submit pull request

## 📖 Additional Resources

- [Svelte 5 Documentation](https://svelte.dev/docs)
- [Vite Documentation](https://vitejs.dev/)
- [Tailwind CSS](https://tailwindcss.com/)
- [DaisyUI Components](https://daisyui.com/)
- [Vitest Testing](https://vitest.dev/)

---

**Last Updated**: $(date)
**Version**: $(cat package.json | jq -r .version)
**Build ID**: $(cat src/lib/build-info.ts | grep buildId | head -1 | sed 's/.*"\(.*\)".*/\1/')
