import { writable, derived } from 'svelte/store';

// Define types for better TypeScript support
type AspectRatio = { id: number; value: string };
export type Screen = { id: number; name: string; media?: string; width?: string };
type Format = { id: number; format: string };
type Multiplier = { id: number; multiplier: string; quality: string };
type Attribute = { id: number; name: string; value: any };
type ViewMode = { 
    id: string; 
    key: string; 
    sizes: any[]; 
    label: string; 
    attributes: Attribute[]; 
    fallback_transform: string 
};

type DrupalSettings = {
    app_config: {
        formats: Record<string, any>;
        media_source: Record<string, any>;
        source: Record<string, any>;
		aspect_ratios: Record<string, any>;
    };
    aspect_ratios: any[];
    source: string;
    media_source: string;
    config: {
        screens: Record<string, any>;
        transforms: string;
        multipliers: any[];
        quality: Record<string, any>;
        formats: any[];
        attributes: Record<string, any>;
        fallback_transform: string;
    };
    view_modes: any;
};

export let aspect_ratios = writable<AspectRatio[]>([]);
export let viewModes = writable<ViewMode[]>([]);
export let screens = writable<Screen[]>([]);
export let formats = writable<Format[]>([]);
export let multipliers = writable<Multiplier[]>([]);
export let attributes = writable<Attribute[]>([]);
export let isSyncing = writable<boolean>(false);
export let isScreensDirty = writable<boolean>(false);
export const isFormValid = writable<boolean>(true); 

export const defaultEmptyConfig = {
	source: "image",
	media_source: "combined_image_style",
	aspect_ratios: [],
	config: {
		screens: {},
		transforms: "",
		multipliers: [],
		quality: {},
		formats: [],
		attributes: {},
		fallback_transform: ""
	},
	view_modes: {}
};

export let drupalSettings = writable<DrupalSettings>({
	app_config: {
		formats: {},
		media_source: {},
		source: {},
		aspect_ratios: {},
	},
	aspect_ratios: [],
	source: "image",
	media_source: "combined_image_style",
	config: {
		screens: {},
		transforms: "",
		multipliers: [],
		quality: {},
		formats: [],
		attributes: {},
		fallback_transform: ""
	},
	view_modes: []
});

function transformViewModeSizes(screenValues: Screen[], viewMode: any) {
	let temAspectRatios = viewMode.aspect_ratios
		.split(/[\s]+/);
	let tempSizes: Record<string, any> = {};
	viewMode.sizes
		.split(/[\s]+/)
		.map((val: string, ndx: number, arr: string[]) => {
			let values = val.split(':');
			tempSizes[values[0]] = {
				width: values[1],
				aspect_ratio: temAspectRatios[ndx]
			};
		});
	let output: any[] = [];
	output = screenValues.map((v: Screen, index: number, arr: Screen[]) => {
		return {
			id: index,
			screen_name: v.name,
			screen_width: v.width,
			enabled: !!tempSizes[v.name],
			width: tempSizes[v.name]?.width,
			aspect_ratio: tempSizes[v.name]?.aspect_ratio,
		};
	});
	return output;
}

function transformAttributes(attributes: Record<string, any>): Attribute[] {
	let attributes_items: Attribute[] = [];
	let id = 0;
	for (const [key, value] of Object.entries(attributes)) {
		attributes_items.push({
			id: id,
			name: key,
			value
		});
		id++;
	}
	return attributes_items;
}

export const transformViewModesToStore = function (screenValues: Screen[], viewModeConfig: Record<string, any>) {
	let viewModesItems: ViewMode[] = [];
	for (const [key, value] of Object.entries(viewModeConfig)) {
		viewModesItems.push({
			id: key,
			key: key,
			sizes: transformViewModeSizes(screenValues, value),
			label: value.label ?? "",
			attributes: transformAttributes(value?.attributes ?? {}),
			fallback_transform: value.fallback_transform ?? "",
		});
	}
	viewModes.set(viewModesItems);
}

export const transformAspectRatiosToStore = function (aspect_ratio_config: string[]) {
	aspect_ratios.set(aspect_ratio_config.map((value: string, ndx: number) => {
		return {
			id: ndx,
			value: value,
		}
	}));
}

export const transformConfigToStore = function (config: any) {
	// Initialize "formats" store.
	multipliers.set(config.multipliers.map((value: string, ndx: number) => {
		return {
			id: ndx,
			multiplier: value,
			quality: config.quality[value] ?? '',
		};
	}));

	formats.set(config.formats.map((value: string, ndx: number) => {
		return {
			id: ndx,
			format: value,
		}
	}))

	let screen_items: Screen[] = [];
	let id = 0;
	for (const [key, value] of Object.entries(config.screens)) {
		screen_items.push({
			id: id,
			name: key,
			media: (value as any).media,
			width: (value as any).width,
		});
		id++;
	}
	screens.set(screen_items);
	attributes.set(transformAttributes(config.attributes));
}