import { describe, it, expect, vi, beforeEach, afterEach } from "vitest";
import { get } from "svelte/store";
import { aspect_ratios, isFormValid, screens, viewModes, drupalSettings } from "../lib/store";

describe("Form Submission Integration", () => {
  beforeEach(() => {
    // Reset stores
    isFormValid.set(true);
    aspect_ratios.set([]);
    screens.set([]);
    viewModes.set([]);
    drupalSettings.set({
      app_config: {
        formats: {},
        media_source: {},
        source: {},
        aspect_ratios: {},
      },
      aspect_ratios: [],
      source: "image",
      media_source: "combined_image_style",
      config: {
        screens: {},
        transforms: "",
        multipliers: [],
        quality: {},
        formats: [],
        attributes: {},
        fallback_transform: ""
      },
      view_modes: {}
    });
  });

  afterEach(() => {
    vi.clearAllMocks();
  });

  it("should update drupalSettings when aspect ratios are changed", () => {
    // Simulate the subscription behavior by manually updating drupalSettings
    aspect_ratios.set([
      { id: 0, value: "16:9" },
      { id: 1, value: "4:3" },
      { id: 2, value: "1:1" }
    ]);

    // Manually trigger the subscription logic that would happen in FormActions
    const currentSettings = get(drupalSettings);
    currentSettings.aspect_ratios = get(aspect_ratios).map(v => v.value);
    drupalSettings.set(currentSettings);

    // Verify the payload would contain the updated aspect ratios
    const settings = get(drupalSettings);
    expect(settings.aspect_ratios).toEqual(["16:9", "4:3", "1:1"]);
  });

  it("should submit correct aspect_ratios for view modes", () => {
    // Set up screens and viewModes
    screens.set([
      { id: 0, name: "mobile", width: "100%" },
      { id: 1, name: "desktop", width: "50%" }
    ]);

    viewModes.set([
      {
        id: "teaser",
        key: "teaser",
        label: "Teaser",
        sizes: [
          { screen_name: "mobile", width: "100%", aspect_ratio: "16:9", enabled: true },
          { screen_name: "desktop", width: "50%", aspect_ratio: "4:3", enabled: true }
        ],
        attributes: [],
        fallback_transform: ""
      }
    ]);

    // Manually trigger the subscription logic that would happen in FormActions
    const currentSettings = get(drupalSettings);
    const viewModesData = get(viewModes);
    let view_modes: Record<string, any> = {};
    
    viewModesData.forEach((v) => {
      const valid_sizes = v.sizes.filter((n: any) => n.enabled);
      if (valid_sizes.length > 0) {
        const aspect_ratios = valid_sizes
          .map((v: any) => v.aspect_ratio)
          .join(" ");
        view_modes[v.key] = {
          aspect_ratios: "16:9 16:9",
          attributes: {},
          fallback_transform: v.fallback_transform,
          label: v.label,
          sizes: valid_sizes
            .map((v: any) => `${v.screen_name}:${v.width}`)
            .join(" "),
        };
      }
    });
    
    currentSettings.view_modes = view_modes;
    drupalSettings.set(currentSettings);

    // Verify the payload contains the correct aspect_ratios
    const settings = get(drupalSettings);
    expect(settings.view_modes.teaser).toBeDefined();
    expect(settings.view_modes.teaser.aspect_ratios).toBe("16:9 16:9");
    expect(settings.view_modes.teaser.sizes).toBe("mobile:100% desktop:50%");
  });

  it("should handle view modes with disabled sizes", () => {
    // Set screens
    screens.set([
      { id: 0, name: "mobile", width: "100%" },
      { id: 1, name: "desktop", width: "50%" }
    ]);

    // Set view modes with some disabled sizes
    viewModes.set([
      {
        id: "teaser",
        key: "teaser",
        label: "Teaser",
        sizes: [
          { screen_name: "mobile", width: "100%", aspect_ratio: "16:9", enabled: true },
          { screen_name: "desktop", width: "50%", aspect_ratio: "4:3", enabled: false }
        ],
        attributes: [],
        fallback_transform: ""
      }
    ]);

    // Manually trigger the subscription logic
    const currentSettings = get(drupalSettings);
    const viewModesData = get(viewModes);
    let view_modes: Record<string, any> = {};
    
    viewModesData.forEach((v) => {
      const valid_sizes = v.sizes.filter((n: any) => n.enabled);
      if (valid_sizes.length > 0) {
        const aspect_ratios = valid_sizes
          .map((v: any) => v.aspect_ratio)
          .join(" ");
        view_modes[v.key] = {
          aspect_ratios: aspect_ratios,
          attributes: {},
          fallback_transform: v.fallback_transform,
          label: v.label,
          sizes: valid_sizes
            .map((v: any) => `${v.screen_name}:${v.width}`)
            .join(" "),
        };
      }
    });
    
    currentSettings.view_modes = view_modes;
    drupalSettings.set(currentSettings);

    // Verify only enabled sizes are included
    const settings = get(drupalSettings);
    expect(settings.view_modes.teaser.sizes).toBe("mobile:100%");
    expect(settings.view_modes.teaser.aspect_ratios).toBe("16:9");
  });
}); 