import { describe, it, expect, beforeEach } from "vitest";
import { get } from "svelte/store";
import {
  drupalSettings,
  aspect_ratios,
  viewModes,
  screens,
  formats,
  multipliers,
  attributes,
  isFormValid,
  transformAspectRatiosToStore,
  transformConfigToStore,
  transformViewModesToStore
} from "../lib/store";

describe("Store Tests", () => {
  beforeEach(() => {
    // Reset all stores to initial state
    drupalSettings.set({
      app_config: {
        formats: {},
        media_source: {},
        source: {},
        aspect_ratios: {},
      },
      aspect_ratios: [],
      source: "image",
      media_source: "combined_image_style",
      config: {
        screens: {},
        transforms: "",
        multipliers: [],
        quality: {},
        formats: [],
        attributes: {},
        fallback_transform: ""
      },
      view_modes: []
    });

    aspect_ratios.set([]);
    viewModes.set([]);
    screens.set([]);
    formats.set([]);
    multipliers.set([]);
    attributes.set([]);
    isFormValid.set(true);
  });

  describe("Store Transformations", () => {
    it("should transform aspect ratios correctly", () => {
      const testAspectRatios = ["16:9", "4:3", "1:1"];
      transformAspectRatiosToStore(testAspectRatios);
      
      const storeValue = get(aspect_ratios);
      expect(storeValue).toHaveLength(3);
      expect(storeValue[0]).toEqual({ id: 0, value: "16:9" });
      expect(storeValue[1]).toEqual({ id: 1, value: "4:3" });
      expect(storeValue[2]).toEqual({ id: 2, value: "1:1" });
    });

    it("should transform config to stores correctly", () => {
      const testConfig = {
        multipliers: ["1x", "2x"],
        quality: { "1x": "80", "2x": "90" },
        formats: ["webp", "jpg"],
        screens: {
          mobile: { media: "(max-width: 768px)", width: "100%" },
          desktop: { media: "(min-width: 769px)", width: "50%" }
        },
        attributes: { alt: "test", title: "test title" }
      };

      transformConfigToStore(testConfig);

      expect(get(multipliers)).toEqual([
        { id: 0, multiplier: "1x", quality: "80" },
        { id: 1, multiplier: "2x", quality: "90" }
      ]);

      expect(get(formats)).toEqual([
        { id: 0, format: "webp" },
        { id: 1, format: "jpg" }
      ]);

      expect(get(screens)).toEqual([
        { id: 0, name: "mobile", media: "(max-width: 768px)", width: "100%" },
        { id: 1, name: "desktop", media: "(min-width: 769px)", width: "50%" }
      ]);

      expect(get(attributes)).toEqual([
        { id: 0, name: "alt", value: "test" },
        { id: 1, name: "title", value: "test title" }
      ]);
    });

    it("should transform view modes correctly", () => {
      const testScreens = [
        { id: 0, name: "mobile", width: "100%" },
        { id: 1, name: "desktop", width: "50%" }
      ];

      const testViewModes = {
        teaser: {
          label: "Teaser",
          sizes: "mobile:100% desktop:50%",
          aspect_ratios: "16:9 4:3",
          fallback_transform: "scale_100",
          attributes: { alt: "teaser alt" }
        }
      };

      transformViewModesToStore(testScreens, testViewModes);
      
      const storeValue = get(viewModes);
      expect(storeValue).toHaveLength(1);
      expect(storeValue[0].key).toBe("teaser");
      expect(storeValue[0].label).toBe("Teaser");
      expect(storeValue[0].sizes).toHaveLength(2);
      expect(storeValue[0].attributes).toHaveLength(1);
    });
  });

  describe("Store Data Validation", () => {
    it("should handle empty aspect ratios", () => {
      transformAspectRatiosToStore([]);
      const storeValue = get(aspect_ratios);
      expect(storeValue).toEqual([]);
    });

    it("should handle empty config", () => {
      const emptyConfig = {
        multipliers: [],
        quality: {},
        formats: [],
        screens: {},
        attributes: {}
      };

      transformConfigToStore(emptyConfig);

      expect(get(multipliers)).toEqual([]);
      expect(get(formats)).toEqual([]);
      expect(get(screens)).toEqual([]);
      expect(get(attributes)).toEqual([]);
    });

    it("should handle empty view modes", () => {
      const testScreens = [
        { id: 0, name: "mobile", width: "100%" }
      ];

      transformViewModesToStore(testScreens, {});
      
      const storeValue = get(viewModes);
      expect(storeValue).toEqual([]);
    });

    it("should handle view modes with missing properties", () => {
      const testScreens = [
        { id: 0, name: "mobile", width: "100%" }
      ];

      const testViewModes = {
        teaser: {
          label: "Teaser",
          sizes: "mobile:100%",
          aspect_ratios: "16:9",
          // Missing fallback_transform and attributes
        }
      };

      transformViewModesToStore(testScreens, testViewModes);
      
      const storeValue = get(viewModes);
      expect(storeValue).toHaveLength(1);
      expect(storeValue[0].fallback_transform).toBe("");
      expect(storeValue[0].attributes).toEqual([]);
    });
  });

  describe("Store State Management", () => {
    it("should initialize stores with correct default values", () => {
      expect(get(aspect_ratios)).toEqual([]);
      expect(get(viewModes)).toEqual([]);
      expect(get(screens)).toEqual([]);
      expect(get(formats)).toEqual([]);
      expect(get(multipliers)).toEqual([]);
      expect(get(attributes)).toEqual([]);
      expect(get(isFormValid)).toBe(true);
    });

    it("should update stores correctly", () => {
      // Test aspect_ratios store
      aspect_ratios.set([{ id: 0, value: "16:9" }]);
      expect(get(aspect_ratios)).toEqual([{ id: 0, value: "16:9" }]);

      // Test multipliers store
      multipliers.set([{ id: 0, multiplier: "1x", quality: "80" }]);
      expect(get(multipliers)).toEqual([{ id: 0, multiplier: "1x", quality: "80" }]);

      // Test screens store
      screens.set([{ id: 0, name: "mobile", width: "100%" }]);
      expect(get(screens)).toEqual([{ id: 0, name: "mobile", width: "100%" }]);

      // Test formats store
      formats.set([{ id: 0, format: "webp" }]);
      expect(get(formats)).toEqual([{ id: 0, format: "webp" }]);

      // Test attributes store
      attributes.set([{ id: 0, name: "alt", value: "test" }]);
      expect(get(attributes)).toEqual([{ id: 0, name: "alt", value: "test" }]);

      // Test viewModes store
      viewModes.set([{ 
        id: "teaser", 
        key: "teaser", 
        label: "Teaser", 
        sizes: [], 
        attributes: [], 
        fallback_transform: "" 
      }]);
      expect(get(viewModes)).toHaveLength(1);
      expect(get(viewModes)[0].key).toBe("teaser");

      // Test isFormValid store
      isFormValid.set(false);
      expect(get(isFormValid)).toBe(false);
    });
  });

  describe("Data Transformation Edge Cases", () => {
    it("should handle malformed aspect ratios", () => {
      const malformedAspectRatios = ["16:9", "", "4:3", null as any, undefined as any];
      transformAspectRatiosToStore(malformedAspectRatios);
      
      const storeValue = get(aspect_ratios);
      expect(storeValue).toHaveLength(5);
      expect(storeValue[0]).toEqual({ id: 0, value: "16:9" });
      expect(storeValue[1]).toEqual({ id: 1, value: "" });
      expect(storeValue[2]).toEqual({ id: 2, value: "4:3" });
      expect(storeValue[3]).toEqual({ id: 3, value: null });
      expect(storeValue[4]).toEqual({ id: 4, value: undefined });
    });

    it("should handle config with missing quality settings", () => {
      const testConfig = {
        multipliers: ["1x", "2x"],
        quality: { "1x": "80" }, // Missing quality for "2x"
        formats: ["webp"],
        screens: {},
        attributes: {}
      };

      transformConfigToStore(testConfig);

      expect(get(multipliers)).toEqual([
        { id: 0, multiplier: "1x", quality: "80" },
        { id: 1, multiplier: "2x", quality: "" } // Should default to empty string
      ]);
    });

    it("should handle view modes with mismatched sizes and aspect ratios", () => {
      const testScreens = [
        { id: 0, name: "mobile", width: "100%" },
        { id: 1, name: "desktop", width: "50%" }
      ];

      const testViewModes = {
        teaser: {
          label: "Teaser",
          sizes: "mobile:100%", // Only one size
          aspect_ratios: "16:9 4:3", // But two aspect ratios
          fallback_transform: "scale_100",
          attributes: {}
        }
      };

      transformViewModesToStore(testScreens, testViewModes);
      
      const storeValue = get(viewModes);
      expect(storeValue).toHaveLength(1);
      expect(storeValue[0].sizes).toHaveLength(2); // Should still have 2 sizes
      // The second size should have undefined aspect_ratio
      expect(storeValue[0].sizes[0].aspect_ratio).toBe("16:9");
      expect(storeValue[0].sizes[1].aspect_ratio).toBeUndefined();
    });
  });
}); 