<?php

namespace Drupal\rift\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\EntityReferenceFieldItemListInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\media\MediaInterface;

/**
 * Plugin implementation of the 'RiftPicture' formatter.
 *
 * @FieldFormatter(
 *   id = "rift_image_entity_reference_picture",
 *   label = @Translation("Rift Image Picture"),
 *   field_types = {
 *     "image"
 *   }
 * )
 */
#[FieldFormatter(
  id: 'rift_image_entity_reference_picture',
  label: new TranslatableMarkup('Rift Image Picture'),
  field_types: ['image'],
)
]
class RiftImagePictureFormatter extends RiftPictureFormatterBase {

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $element = [];
    $view_mode = $this->getSetting('view_mode');
    if (!($items instanceof EntityReferenceFieldItemListInterface)) {
      return $element;
    }
    $media = $items->getParent()->getEntity();
    assert($media instanceof MediaInterface);
    foreach ($this->getEntitiesToView($items, $langcode) as $delta => $entity) {
      $element[$delta] = $this->buildRenderable($media, $view_mode);
    }
    return $element;
  }


  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
        'view_mode' => '',
      ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements['view_mode'] = [
      '#type' => 'select',
      // @todo Convert Rift Media options to plugins/settings.
      '#options' => $this->getAvailableViewModes(),
      '#title' => $this->t('Rift View mode'),
      '#default_value' => $this->getSetting('view_mode'),
      '#required' => TRUE,
    ];
    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    if (!parent::isApplicable($field_definition)
      || $field_definition->getTargetEntityTypeId() !== 'media') {
      return FALSE;
    }
    return TRUE;
  }
}
