<?php

namespace Drupal\rift\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\EntityReferenceFieldItemListInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\media\Entity\MediaType;
use Drupal\media\MediaInterface;

/**
 * Plugin implementation of the 'RiftPicture' formatter.
 *
 * @FieldFormatter(
 *   id = "rift_media_entity_reference_picture",
 *   label = @Translation("Rift Media Picture"),
 *   field_types = {
 *     "entity_reference"
 *   }
 * )
 */
#[FieldFormatter(
  id: 'rift_media_entity_reference_picture',
  label: new TranslatableMarkup('Rift Media Picture'),
  field_types: ['entity_reference'],
)
]
class RiftMediaPictureFormatter extends RiftPictureFormatterBase {

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $element = [];
    $view_mode = $this->getSetting('view_mode');
    if (!($items instanceof EntityReferenceFieldItemListInterface)) {
      return $element;
    }

    foreach ($this->getEntitiesToView($items, $langcode) as $delta => $entity) {
      assert($entity instanceof MediaInterface);
      $element[$delta] = $this->buildRenderable($entity, $view_mode);
    }
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
        'view_mode' => '',
        'link' => FALSE,
      ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements['view_mode'] = [
      '#type' => 'select',
      // @todo Convert Rift Media options to plugins/settings.
      '#options' => $this->getAvailableViewModes(),
      '#title' => $this->t('Rift View mode'),
      '#default_value' => $this->getSetting('view_mode'),
      '#required' => TRUE,
    ];
    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    if (!parent::isApplicable($field_definition)
      || $field_definition->getType() !== 'entity_reference'
      || $field_definition->getSetting('target_type') !== 'media') {
      return FALSE;
    }
    $handler_settings = $field_definition->getSetting('handler_settings');
    if (!isset($handler_settings['target_bundles'])) {
      return FALSE;
    }
    $target_bundles_image = TRUE;
    foreach ($handler_settings['target_bundles'] as $target_bundle) {
      $media_type = MediaType::load($target_bundle);
      if ($media_type->getSource()->getPluginId() !== 'image') {
        $target_bundles_image = FALSE;
        break;
      }
    }
    return $target_bundles_image;
  }
}
