<?php

namespace Drupal\rift\Plugin\Field\FieldFormatter;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\EntityReferenceFieldItemListInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\EntityReferenceEntityFormatter;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\media\Entity\Media;
use Drupal\media\MediaInterface;
use Drupal\rift\RiftFormatterTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'RiftPicture' formatter.
 *
 * @FieldFormatter(
 *   id = "rift_media_entity_reference_picture_with_fallback",
 *   label = @Translation("Rift Media Picture (with fallback)"),
 *   field_types = {
 *     "entity_reference"
 *   }
 * )
 */
#[FieldFormatter(
  id: 'rift_media_entity_reference_picture_with_fallback',
  label: new TranslatableMarkup('Rift Media Picture (with fallback)'),
  field_types: ['entity_reference'],
)
]
class RiftMediaPictureWithFallbackFormatter extends RiftPictureFormatterBase {

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements['rift_view_mode'] = [
      '#type' => 'select',
      '#options' => $this->getAvailableViewModes(),
      '#title' => $this->t('Rift View mode'),
      '#default_value' => $this->getSetting('view_mode'),
      '#required' => TRUE,
    ];
    $elements['view_mode'] = [
      '#type' => 'select',
      '#options' => $this->entityDisplayRepository->getViewModeOptions($this->getFieldSetting('target_type')),
      '#title' => $this->t('View Mode (Fallback)'),
      '#default_value' => $this->getSetting('view_mode'),
      '#required' => TRUE,
      '#description' => $this->t('Set setting is identical to the "View Mode" setting in Entity Reference formmater. \
      In case the target media bundle is not compatible with RIFT (E.g. Video, SVG, etc.) then the entity would be rendered\
      using the selected view mode.'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
        'rift_view_mode' => '',
        'view_mode' => 'default',
        'link' => FALSE,
      ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = parent::viewElements($items, $langcode);
    $view_mode = $this->getSetting('rift_view_mode');
    if (!($items instanceof EntityReferenceFieldItemListInterface)) {
      return $elements;
    }
    foreach ($this->getEntitiesToView($items, $langcode) as $delta => $entity) {
      assert($entity instanceof MediaInterface);
      if ($entity->getSource()->getPluginId() !== 'image') {
        continue;
      }
      $elements[$delta] = $this->buildRenderable($entity, $view_mode);
    }
    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    if (parent::isApplicable($field_definition) &&
      $field_definition->getType() == 'entity_reference' &&
      $field_definition->getSetting('target_type') == 'media') {
      return TRUE;
    }
    return FALSE;
  }

}
